<?php
require '../tpst/db.php';

// Set timezone and 30-day window
date_default_timezone_set('America/Chicago');
$today = new DateTime();
$end = (clone $today)->modify('+30 days');

$start_str = $today->format('Y-m-d');
$end_str = $end->format('Y-m-d');

// Get all available days
$availability_stmt = $conn->prepare("
    SELECT date FROM TestDayAvailability 
    WHERE is_available = 1 AND date BETWEEN ? AND ?
    ORDER BY date ASC
");
$availability_stmt->bind_param("ss", $start_str, $end_str);
$availability_stmt->execute();
$availability_result = $availability_stmt->get_result();

$available_dates = [];
while ($row = $availability_result->fetch_assoc()) {
    $available_dates[$row['date']] = 0; // initialize with 0 appointments
}
$availability_stmt->close();

// Count how many test appointments exist on each available day
if (!empty($available_dates)) {
    $placeholders = implode(',', array_fill(0, count($available_dates), '?'));
    $types = str_repeat('s', count($available_dates));
    $values = array_keys($available_dates);

    $appt_stmt = $conn->prepare("
        SELECT test_date, COUNT(*) AS num_appts
        FROM TestAppointments
        WHERE test_date IN ($placeholders)
        GROUP BY test_date
    ");
    $appt_stmt->bind_param($types, ...$values);
    $appt_stmt->execute();
    $appt_result = $appt_stmt->get_result();

    while ($row = $appt_result->fetch_assoc()) {
        $available_dates[$row['test_date']] = $row['num_appts'];
    }
    $appt_stmt->close();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>TPST 30-Day Calendar</title>
    <style>
        body {
            background: linear-gradient(to right, #2c3e50, #4b6584);
            font-family: Arial, sans-serif;
            padding: 40px;
            color: #333;
        }
        .container {
            max-width: 1100px;
            margin: auto;
            background: linear-gradient(to right, #ffffff, #f0f4f8);
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 0 12px rgba(0,0,0,0.1);
        }
        .calendar-columns {
            display: flex;
            gap: 40px;
            justify-content: space-between;
            flex-wrap: wrap;
        }
        .column {
            flex: 1;
            min-width: 400px;
        }
        .day-block {
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 1px dashed #ccc;
        }
        
        .date-header {
            font-weight: bold;
            color: #2c3e50;
            margin-bottom: 6px;
        }
        
        .slot {
            display: inline-block;
            background: #d5f5e3;
            border-radius: 6px;
            padding: 6px 10px;
            margin: 3px;
            font-size: 0.95em;
        }
        
        .no-slots {
            font-style: italic;
            color: #aaa;
        }
        h2 {
            text-align: center;
            margin-bottom: 30px;
            color: white;
        }
        .main-content {
            display: flex;
            justify-content: space-between;
            gap: 30px;
        }
        .calendar {
            flex: 3;
        }
        .sidebar {
            flex: 1;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: flex-start;
        }
        .book-btn-inline {
            display: inline-block;
            margin-left: 10px;
            padding: 6px 12px;
            background-color: #3498db;
            color: white;
            font-weight: bold;
            border-radius: 6px;
            text-decoration: none;
            font-size: 0.85em;
        }
        .book-btn-inline:hover {
            background-color: #2c80b4;
}

        .book-btn {
            padding: 16px 30px;
            font-size: 18px;
            background-color: #27ae60;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: bold;
            margin-top: 20px;
            transition: background-color 0.3s ease;
        }
        .book-btn:hover {
            background-color: #1e8449;
        }
        table {
            width: 100%;
            border-collapse: collapse;
        }
        td {
            padding: 8px 4px;
            vertical-align: top;
        }
        .date-header {
            font-weight: bold;
            padding-top: 18px;
            color: white;
        }
        .slot {
            margin: 4px 4px 4px 0;
            padding: 6px 10px;
            display: inline-block;
            background: #d5f5e3;
            border-radius: 5px;
            font-size: 0.95em;
        }
        .full {
            background: #f9cfcf;
            color: #c0392b;
        }
        .no-slots {
            font-style: italic;
            color: #aaa;
        }
        @media screen and (max-width: 900px) {
        .calendar-columns {
            flex-direction: column;
        }
    </style>
</head>
<body>
        <div class="calendar">
    <div class="calendar-columns">
        <?php for ($col = 0; $col < 2; $col++): ?>
            <div class="column">
                <?php for ($i = $col * 15; $i < ($col + 1) * 15; $i++): ?>
                    <?php
                        $date = (clone $today)->modify("+{$i} days")->format('Y-m-d');
                        $weekday = date('l', strtotime($date));
                    ?>
                    <div class="day-block">
                        <div class="date-header">
                            <?= $weekday ?>, <?= date('F j, Y', strtotime($date)) ?>
                            <a href="book-from-cal.php?date=<?= $date ?>" class="book-btn-inline">Book</a>
                        </div>
                        <div class="slots">
                            <?php if (isset($available_dates[$date])): ?>
                                <?php
                                    $count = $available_dates[$date];
                                    if ($count >= 7): ?>
                                        <span class="slot full">Full (7/7)</span>
                                    <?php else: ?>
                                        <span class="slot"><?= 7 - $count ?> slots left</span>
                                    <?php endif; ?>
                            <?php else: ?>
                                <span class="no-slots">Unavailable</span>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endfor; ?>
            </div>
        <?php endfor; ?>
    </div>
</div>
    </div>
</div>
</body>
</html>


