<?php
session_start();
require_once __DIR__ . '/../tpst/db.php';

// ✅ Admin has full access
if (isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true) {
    $features = [
        'can_upload_documents' => true,
        'can_access_log_testing' => true
    ];
} else {
    // ✅ Subscriber fallback
    require_once __DIR__ . '/includes/tier-config.php';
    $tier = $_SESSION['tier_choice'] ?? 'basic';
    $features = $tier_permissions[$tier] ?? $tier_permissions['basic'];

    // If log-testing not allowed, bounce
    if (empty($features['can_access_log_testing'])) {
        header("Location: /tpst/no-doc-access.php");
        exit;
    }
}

// Continue normal execution for allowed user

$today = date("Y-m-d");
$earlier = date("Y-m-d", strtotime("-14 days"));

// Fetch appointments for dropdown
$appts = $conn->query("SELECT id, full_name, test_date FROM TestAppointments WHERE test_date BETWEEN '$earlier' AND '$today' ORDER BY test_date DESC");

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $appointment_id = intval($_POST['appointment_id']);
    $result = $_POST['result'];
    $notes = trim($_POST['notes']);

    // Create folder for uploaded packet
    $folder_name = "test-packets/" . date("Ymd") . "-appt-$appointment_id";
    $full_path = __DIR__ . "/$folder_name";
    if (!is_dir($full_path)) {
        mkdir($full_path, 0775, true);
    }

    // Insert log entry
    $stmt = $conn->prepare("INSERT INTO TestLogs (appointment_id, test_date, result, examiner_notes, packet_path) VALUES (?, CURDATE(), ?, ?, ?)");
    $stmt->bind_param("isss", $appointment_id, $result, $notes, $folder_name);
    $stmt->execute();
    $stmt->close();

    // Save uploaded files to folder (up to 2 per field)
    $upload_fields = ['de_964', 'impact', 'dl_40', 'dl_14a', 'dl_14b', 'other'];
    foreach ($upload_fields as $field) {
        if (!empty($_FILES[$field]['name'][0])) {
            for ($i = 0; $i < count($_FILES[$field]['name']) && $i < 2; $i++) {
                $tmp = $_FILES[$field]['tmp_name'][$i];
                $original = basename($_FILES[$field]['name'][$i]);
                $stored = uniqid() . "-" . preg_replace('/\s+/', '_', $original);
                $dest = "$full_path/$stored";

                if (move_uploaded_file($tmp, $dest)) {
                    // Files are saved, but not recorded individually in DB
                    // If needed, consider adding a 'TestLogFiles' table later
                }
            }
        }
    }

    echo "<p style='color: green; font-weight: bold;'>✅ Test log saved and files uploaded successfully!</p>";
}
?>


<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Log Test Results</title>
    <style>
        body { font-family: Arial, sans-serif; background: #f0f0f0; padding: 30px; }
        .container { background: white; padding: 30px; border-radius: 10px; max-width: 800px; margin: auto; box-shadow: 0 0 10px rgba(0,0,0,0.2); }
        h2 { text-align: center; }
        label { font-weight: bold; display: block; margin-top: 15px; }
        input[type="file"] { margin-top: 5px; margin-bottom: 10px; }
        textarea { width: 100%; height: 80px; margin-top: 5px; }
        select, textarea { padding: 8px; border-radius: 5px; border: 1px solid #ccc; width: 100%; }
        .submit-btn {
            background: #2ecc71;
            color: white;
            font-weight: bold;
            padding: 12px 20px;
            border: none;
            border-radius: 6px;
            margin-top: 20px;
            cursor: pointer;
            width: 100%;
        }
        .submit-btn:hover {
            background: #27ae60;
        }
    </style>
</head>
<body>
    <div class="container">
        <h2>Log Test Results</h2>
        <form action="log-test-results.php" method="POST" enctype="multipart/form-data">
            <label for="appointment_id">Select Appointment:</label>
            <select name="appointment_id" id="appointment_id" required>
                <option value="">-- Select Appointment --</option>
                <?php
                $twoWeeksAgo = date('Y-m-d', strtotime('-14 days'));
                $stmt = $conn->prepare("SELECT id, full_name, test_date FROM TestAppointments WHERE test_date >= ? ORDER BY test_date DESC");
                $stmt->bind_param("s", $twoWeeksAgo);
                $stmt->execute();
                $result = $stmt->get_result();
                while ($row = $result->fetch_assoc()):
                ?>
                    <option value="<?= $row['id'] ?>">
                        <?= htmlspecialchars($row['full_name']) ?> (<?= $row['test_date'] ?>)
                    </option>
                <?php endwhile;
                $stmt->close(); ?>
            </select>

            <label>Result:</label>
            <label><input type="radio" name="result" value="pass" required> Pass</label>
            <label><input type="radio" name="result" value="fail" required> Fail</label>

            <label for="notes">Examiner Notes:</label>
            <textarea name="notes" placeholder="Optional notes about test performance, conditions, etc."></textarea>

            <label>Upload DE-964 Certificates (max 2):</label>
            <input type="file" name="de_964[]" multiple accept=".pdf,.jpg,.jpeg,.png">

            <label>Upload IMPACT Certificates (max 2):</label>
            <input type="file" name="impact[]" multiple accept=".pdf,.jpg,.jpeg,.png">

            <label>Upload DL-40 Form(s) (max 2):</label>
            <input type="file" name="dl_40[]" multiple accept=".pdf,.jpg,.jpeg,.png">

            <label>Upload DL-14A Form(s) (max 2):</label>
            <input type="file" name="dl_14a[]" multiple accept=".pdf,.jpg,.jpeg,.png">

            <label>Upload DL-14B Form(s) (max 2):</label>
            <input type="file" name="dl_14b[]" multiple accept=".pdf,.jpg,.jpeg,.png">

            <label>Other Files (max 2):</label>
            <input type="file" name="other[]" multiple>

            <button type="submit" class="submit-btn">Submit Log & Upload Files</button>
        </form>
        <br>
<a href="testing.php" style="
    display: inline-block;
    text-align: center;
    background: #3498db;
    color: white;
    font-weight: bold;
    padding: 12px 20px;
    border-radius: 6px;
    text-decoration: none;
    margin-top: 20px;
    width: 100%;
">⬅️ Back to Testing Dashboard</a>

    </div>
</body>
</html>

