<?php
session_start();
require 'db.php';

// Redirect if not logged in as instructor
if (!isset($_SESSION['instructor_id'])) {
    header("Location: instructor-login.php");
    exit();
}

    $instructor_id = $_SESSION['instructor_id'];
    $success = "";
    $error = "";

    // Get current max car count
    $car_stmt = $conn->prepare("SELECT COUNT(*) AS car_count FROM fleet WHERE status = 'active'");
    $car_stmt->execute();
    $car_result = $car_stmt->get_result();
    $car_row = $car_result->fetch_assoc();
    $max_cars = intval($car_row['car_count']);
    $car_stmt->close();

if ($_SERVER["REQUEST_METHOD"] === "POST") {
    $date = $_POST['date'] ?? '';
    $start_time = $_POST['start_time'] ?? '';
    $end_time = $_POST['end_time'] ?? '';
    $instructor_id = $_SESSION['instructor_id'];

    if (!$date || !$start_time || !$end_time) {
        $error = "❌ Please fill out all fields before submitting.";
    } else {
        $selected_date = new DateTime($date);
        $today = new DateTime();
        $today->setTime(0, 0); 

        if ($selected_date < $today) {
            $error = "You cannot submit availability for a past date.";
        } else {
            $start = new DateTime($start_time);
            $end = new DateTime($end_time);
            $interval = $start->diff($end);
            $duration_in_hours = $interval->h + ($interval->i / 60);

            if ($duration_in_hours <= 0 || $duration_in_hours > 2) {
                $error = "Each availability slot must be 1 or 2 hours only.";
            } else {
                // Check instructor's total hours for the day
                $stmt_total = $conn->prepare("
                    SELECT start_time, end_time 
                    FROM InstructorAvailability 
                    WHERE instructor_id = ? AND available_date = ?
                ");
                $stmt_total->bind_param("is", $instructor_id, $date);
                $stmt_total->execute();
                $result = $stmt_total->get_result();
                $total_hours = 0;
                while ($row = $result->fetch_assoc()) {
                    $s = new DateTime($row['start_time']);
                    $e = new DateTime($row['end_time']);
                    $int = $s->diff($e);
                    $total_hours += $int->h + ($int->i / 60);
                }
                $stmt_total->close();

                if (($total_hours + $duration_in_hours) > 12) {
                    $error = "This slot would exceed your 12-hour daily limit.";
                } else {
                    // Check for overlap with same instructor
                    $stmt = $conn->prepare("
                        SELECT id FROM InstructorAvailability 
                        WHERE instructor_id = ? AND available_date = ? 
                          AND ((start_time < ? AND end_time > ?) 
                               OR (start_time < ? AND end_time > ?))
                    ");
                    $stmt->bind_param("isssss", $instructor_id, $date, $end_time, $start_time, $start_time, $end_time);
                    $stmt->execute();
                    $stmt->store_result();

                    if ($stmt->num_rows > 0) {
                        $error = "You already have availability that overlaps with this time.";
                        $stmt->close();
                    } else {
                        $stmt->close();

                        // Check if more than 2 instructors already scheduled for this exact slot
                        $stmt = $conn->prepare("
                            SELECT COUNT(*) AS count
                            FROM InstructorAvailability
                            WHERE available_date = ?
                              AND start_time = ?
                              AND end_time = ?
                        ");
                        $stmt->bind_param("sss", $date, $start_time, $end_time);
                        $stmt->execute();
                        $result = $stmt->get_result();
                        $row = $result->fetch_assoc();
                        $stmt->close();

                        if ($row['count'] >= $max_cars) {
                            $error = "⚠️ This time slot is already at capacity ({$max_cars} instructors).";
                        } else {
                            // Insert availability
                            $stmt = $conn->prepare("
                                INSERT INTO InstructorAvailability (instructor_id, available_date, start_time, end_time)
                                VALUES (?, ?, ?, ?)
                            ");
                            $stmt->bind_param("isss", $instructor_id, $date, $start_time, $end_time);
                            if ($stmt->execute()) {
                                $success = "✅ Availability submitted successfully.";
                            } else {
                                $error = "❌ Failed to save availability. Please try again.";
                            }
                            $stmt->close();
                        }
                    }
                }
            }
        }
    }
}

// Get all availability for this instructor
$availabilities = [];
$stmt = $conn->prepare("SELECT available_date, start_time, end_time
    FROM InstructorAvailability
    WHERE instructor_id = ?
    AND CONCAT(available_date, ' ', start_time) >= NOW()
    ORDER BY available_date, start_time
    ");
$stmt->bind_param("i", $instructor_id);
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()) {
    $availabilities[] = $row;
}
$stmt->close();
$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Instructor Availability</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            background-color: #f4f4f4;
            padding: 40px;
        }
        .container {
            background: #fff;
            padding: 25px;
            border-radius: 8px;
            max-width: 500px;
            margin: 0 auto;
            box-shadow: 0px 0px 10px rgba(0,0,0,0.1);
        }
        h2 {
            text-align: center;
        }
        label {
            margin-top: 10px;
            display: block;
        }
        input[type="date"],
        input[type="time"] {
            width: 100%;
            padding: 8px;
            margin-top: 5px;
            border-radius: 4px;
            border: 1px solid #ccc;
        }
        input[type="submit"],
        .button-link {
            display: inline-block;
            margin-top: 20px;
            padding: 10px 15px;
            background: #007bff;
            color: white;
            border: none;
            border-radius: 4px;
            text-decoration: none;
            cursor: pointer;
        }
        .button-link:hover {
            background: #0056b3;
        }
        .success { color: green; }
        .error { color: red; }
        table {
            width: 100%;
            margin-top: 25px;
            border-collapse: collapse;
        }
        th, td {
            border: 1px solid #ccc;
            padding: 8px;
            text-align: center;
        }
        th {
            background-color: #eee;
        }
    </style>
</head>
<body>
<div class="container">
    <h2>Submit Weekly Drive Availability</h2>

    <?php if ($success): ?>
        <p class="success"><?php echo $success; ?></p>
        <a href="instructor-availability.php" class="button-link">Submit Another</a>
        <a href="instructor-dashboard.php" class="button-link">Return to Dashboard</a>
    
    <?php else: ?>
        <?php if ($error): ?><p class="error"><?php echo $error; ?></p><?php endif; ?>
        
        <form method="post">
            <label for="date">Date:</label>
            <?php $today = date('Y-m-d'); ?>
            <input type="date" name="date" min="<?php echo $today; ?>" required>

            <label for="start_time">Start Time:</label>
            <input type="time" name="start_time" required>

            <label for="end_time">End Time:</label>
            <input type="time" name="end_time" required>

            <div style="display: flex; gap: 10px; margin-top: 15px;">
                <input type="submit" value="Submit Availability" style="flex: 1; background: #007bff; color: white; border: none; padding: 10px; border-radius: 4px; cursor: pointer;">
            </div>
        </form>

        <!-- Separate form outside the post form -->
        <form method="get" action="instructor-dashboard.php">
            <button type="submit" class="button-link" style="margin-top: 15px; width: 100%; padding: 10px; border-radius: 4px;">
                Return to Dashboard
            </button>
        </form>

    <?php endif; ?>

<?php if (!empty($availabilities)): ?>
    <h3>Your Scheduled Availability</h3>
    <table>
        <?php foreach ($availabilities as $a): ?>
        <tr>
            <td><?= $a['available_date'] ?></td>
            <td><?= $a['start_time'] ?> - <?= $a['end_time'] ?></td>
            <td>
                <form method="post" action="cancel-slot.php" style="margin: 0;">
                    <input type="hidden" name="slot_date" value="<?= $a['available_date'] ?>">
                    <input type="hidden" name="start_time" value="<?= $a['start_time'] ?>">
                    <input type="hidden" name="end_time" value="<?= $a['end_time'] ?>">
                    <button type="submit"
                            onclick="return confirm('Are you sure you want to cancel this availability slot?');"
                            style="background: #dc3545; color: white; border: none; padding: 5px 10px; border-radius: 4px; cursor: pointer;">
                        Cancel
                    </button>
                </form>
            </td>
        </tr>
        <?php endforeach; ?>
    </table>
<?php endif; ?>
    </div>
</body>
</html>



