<?php
require 'db.php'; // Adjust path if needed

$today = date('Y-m-d');
$deleted = false;
$error = '';

try {
    // Step 1: Get IDs of past appointments
    $stmt = $conn->prepare("SELECT id FROM TestAppointments WHERE test_date < ?");
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $result = $stmt->get_result();

    $past_ids = [];
    while ($row = $result->fetch_assoc()) {
        $past_ids[] = $row['id'];
    }
    $stmt->close();

    if (!empty($past_ids)) {
        $conn->begin_transaction();

        $ids_str = implode(',', array_map('intval', $past_ids));

        // Step 1: Delete from TestDocuments (joined via TestLogs)
        $conn->query("DELETE td FROM TestDocuments td 
                      JOIN TestLogs tl ON td.test_log_id = tl.id 
                      WHERE tl.appointment_id IN ($ids_str)");

        // Step 2: Delete from TestLogs
        $conn->query("DELETE FROM TestLogs WHERE appointment_id IN ($ids_str)");

        // Step 3: Delete from TestAppointments
        $conn->query("DELETE FROM TestAppointments WHERE id IN ($ids_str)");

        // Optional: Reset AUTO_INCREMENT values
        $conn->query("ALTER TABLE TestDocuments AUTO_INCREMENT = 1");
        $conn->query("ALTER TABLE TestLogs AUTO_INCREMENT = 1");
        $conn->query("ALTER TABLE TestAppointments AUTO_INCREMENT = 1");

        $conn->commit();
        $deleted = true;
    }

} catch (Exception $e) {
    $conn->rollback();
    $error = "Error clearing past appointments: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>Clear Past Appointments</title>
    <style>
        body {
            background: #2c3e50;
            font-family: Arial, sans-serif;
            color: white;
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100vh;
        }
        .message-box {
            padding: 30px;
            background: #34495e;
            border-radius: 10px;
            box-shadow: 0 0 10px black;
            text-align: center;
        }
        .message-box a {
            color: #ecf0f1;
            display: inline-block;
            margin-top: 20px;
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="message-box">
        <?php if ($deleted): ?>
            <h2>✅ Past test appointments successfully cleared.</h2>
        <?php elseif (!empty($error)): ?>
            <h2>⚠️ <?= htmlspecialchars($error) ?></h2>
        <?php else: ?>
            <h2>ℹ️ No past appointments to clear.</h2>
        <?php endif; ?>
        <a href="testing.php">← Back to Dashboard</a>
    </div>
</body>
</html>
