<?php
require 'db.php';

$today = new DateTime();
$dates = [];

// Pull next 30 testable days (Mon–Sat)
for ($i = 0; count($dates) < 30; $i++) {
    $candidate = (clone $today)->modify("+$i days");
    if (in_array($candidate->format('N'), [1,2,3,4,5,6])) {
        $dates[] = $candidate->format('Y-m-d');
    }
}

// Get availability status
function loadAvailability($conn) {
    $result = $conn->query("SELECT date, is_available FROM TestDayAvailability");
    $availability = [];
    while ($row = $result->fetch_assoc()) {
        $availability[$row['date']] = $row['is_available'];
    }
    return $availability;
}

$availability = loadAvailability($conn);

// Re-fetch accepted and pending dates (outside POST to ensure freshness)
$accepted_dates = [];
$pending_dates = [];

$book_stmt = $conn->prepare("SELECT test_date, status FROM TestAppointments WHERE test_date >= CURDATE() AND test_time < '10:00:00'");
$book_stmt->execute();
$book_stmt->bind_result($test_date, $status);
while ($book_stmt->fetch()) {
    $date_only = date('Y-m-d', strtotime($test_date));
    if ($status === 'accepted') {
        $accepted_dates[] = $date_only;
    } elseif ($status === 'pending') {
        $pending_dates[] = $date_only;
    }
}
$book_stmt->close();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $selected = isset($_POST['available']) ? $_POST['available'] : [];
    $warnings = [];

    foreach ($dates as $date) {
        $is_available = in_array($date, $selected) ? 1 : 0;
        if ($is_available === 0 && in_array($date, $accepted_dates)) {
            $warnings[] = $date;
        }
    }

    if (!empty($warnings) && !isset($_POST['confirm_override'])) {
        echo "<h2 style='color: red;'>\u26a0\ufe0f Warning: You are disabling availability on dates that already have booked appointments.</h2>";
        echo "<ul style='color: red;'>";
        foreach ($warnings as $warn_date) {
            echo "<li>" . date("l, F j", strtotime($warn_date)) . "</li>";
        }
        echo "</ul>";
        echo "<form method='post'>";
        foreach ($selected as $val) {
            echo "<input type='hidden' name='available[]' value='" . htmlspecialchars($val) . "'>";
        }
        echo "<input type='hidden' name='confirm_override' value='1'>";
        echo "<button type='submit' style='background-color: #e67e22; color: white; padding: 10px; border: none; border-radius: 5px;'>Confirm and Save Anyway</button>";
        echo " <a href='toggle-availability.php' style='margin-left: 15px;'>Cancel</a>";
        echo "</form>";
        exit();
    }

    foreach ($dates as $date) {
        $is_available = in_array($date, $selected) ? 1 : 0;

        $stmt = $conn->prepare("INSERT INTO TestDayAvailability (date, is_available) VALUES (?, ?) ON DUPLICATE KEY UPDATE is_available = VALUES(is_available)");
        $stmt->bind_param("si", $date, $is_available);
        $stmt->execute();
        $stmt->close();

        // Delete appointments if toggled off
        if ($is_available === 0) {
            $delete_stmt = $conn->prepare("DELETE FROM TestAppointments WHERE test_date = ?");
            $delete_stmt->bind_param("s", $date);
            $delete_stmt->execute();
            $delete_stmt->close();
        }
    }

    header("Location: toggle-availability.php?saved=1");
    exit();
}
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Toggle Test Day Availability</title>
    <style>
        body {
            font-family: Arial;
            background: linear-gradient(to right, #bdc3c7, #2c3e50);
            padding: 40px 20px;
            min-height: 100vh;
        }

        form {
            background: rgba(255, 255, 255, 0.85);
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 0 10px #ccc;
        }

        .grid-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
            gap: 10px;
        }

        .date-label {
            display: flex;
            align-items: center;
            padding: 8px 12px;
            border-radius: 6px;
            box-shadow: 0 1px 4px rgba(0,0,0,0.1);
            background-color: #f9f9f9;
            transition: background-color 0.2s ease;
        }

        .booked {
            background-color: #fce4e4 !important;
            color: #b30000;
            font-weight: bold;
        }

        .date-label:hover {
            background-color: #e0f7fa;
        }

        input[type="submit"] {
            margin-top: 20px;
            padding: 10px 20px;
            font-weight: bold;
            background-color: #3498db;
            border: none;
            color: white;
            border-radius: 6px;
            cursor: pointer;
        }

        h2 {
            color: white;
            text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.7);
        }

        .date-label.booked {
            background-color: #fce4e4 !important;
            color: #b30000 !important;
            font-weight: bold;
        }

    </style>
</head>
<body>

<h2>Toggle Morning Test Availability (Next 30 Weekdays)</h2>
<?php if (isset($_GET['saved'])): ?>
    <p style="color: green;">Availability updated!</p>
<?php endif; ?>

<form method="post">
    <div class="grid-container">
        <?php foreach ($dates as $date): ?>
            <?php
                $checked = isset($availability[$date]) ? $availability[$date] == 1 : true;
                $formatted = date("l, F j", strtotime($date));
               $is_accepted = in_array($date, $accepted_dates);
                $is_pending  = in_array($date, $pending_dates);
                $label_class = 'date-label';
                
                if ($is_accepted) {
                    $label_class .= ' booked';
                } elseif ($is_pending) {
                    $label_class .= ' pending';
                }
            ?>
            <label class="<?= $label_class ?>">
                <input type="checkbox" name="available[]" value="<?= $date ?>" <?= $checked ? 'checked' : '' ?> style="margin-right: 10px;">
                <?= $formatted ?>
                <?= $is_accepted ? ' (Booked)' : ($is_pending ? ' (Pending)' : '') ?>
            </label>
        <?php endforeach; ?>
    </div>
    <div style="display: flex; justify-content: center; margin-top: 20px;">
        <input type="submit" value="Save Availability">
    </div>
</form>

<div style="text-align: center; margin-bottom: 20px;">
    <form method="post" action="clear-past-appointments.php" onsubmit="return confirm('Are you sure you want to delete all past appointments?');">
        <button type="submit" style="
            background-color: #e74c3c;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            font-weight: bold;
            cursor: pointer;
        ">
            🗑️ Clear Past Appointments
        </button>
    </form>
</div>

<div style="text-align: center; margin-bottom: 20px;">
    <a href="testing.php" style="
        display: inline-block;
        padding: 10px 20px;
        background-color: #2ecc71;
        color: white;
        font-weight: bold;
        text-decoration: none;
        border-radius: 6px;
        box-shadow: 0 2px 6px rgba(0,0,0,0.2);
    ">
        ← Back to TPST Control Panel
    </a>
</div>

</body>
</html>
