<?php
session_start();
date_default_timezone_set('America/Chicago');
require 'db.php';
require_once 'includes/send_email.php';

if (!isset($_SESSION['instructor_id']) && !isset($_SESSION['admin_id'])) {
    header("Location: instructor-login.php");
    exit();
}

$tz = new DateTimeZone('America/Chicago');
$success = "";
$error   = "";

// How many students can ride in one car during a single time slot
const MAX_STUDENTS_PER_SLOT = 3;

/* ---------------------- Helpers ---------------------- */

// Optional setting (global default capacity)
function get_setting(mysqli $conn, string $key) {
    if (!($stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key=? LIMIT 1"))) return null;
    $stmt->bind_param('s', $key);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res ? $res->fetch_assoc() : null;
    if ($res) { $res->free(); }
    $stmt->close();
    return $row ? $row['setting_value'] : null;
}

// Count active cars in fleet (fallback if no overrides/globals)
function active_cars_count(mysqli $conn): int {
    $rs = $conn->query("SELECT COUNT(*) AS c FROM fleet WHERE status='active'");
    $row = $rs ? $rs->fetch_assoc() : null;
    if ($rs) { $rs->free(); }
    return $row ? (int)$row['c'] : 0;
}

// Per-slot override lookup (week/day/time). Safe if table doesn't exist.
function get_capacity_override_for_dt(mysqli $conn, DateTime $dt): ?int {
    static $hasTable = null;
    if ($hasTable === null) {
        $check = $conn->query("SHOW TABLES LIKE 'fleet_capacity_overrides'");
        $hasTable = $check && $check->num_rows > 0;
        if ($check) { $check->free(); }
    }
    if (!$hasTable) return null;

    $date = $dt->format('Y-m-d');
    $time = $dt->format('H:i:s');
    $sql = "
      SELECT capacity,
             IFNULL(TIMESTAMPDIFF(MINUTE, start_time, end_time), 24*60) AS span
      FROM fleet_capacity_overrides
      WHERE override_date = ?
        AND (start_time IS NULL OR start_time <= ?)
        AND (end_time   IS NULL OR end_time   >  ?)
      ORDER BY (start_time IS NULL) ASC, span ASC
      LIMIT 1";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('sss', $date, $time, $time);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res ? $res->fetch_assoc() : null;
    if ($res) { $res->free(); }
    $stmt->close();
    return $row ? (int)$row['capacity'] : null;
}

// Vehicles down at a specific moment (safe if table doesn't exist)
function vehicles_down_count_at(mysqli $conn, DateTime $dt): int {
    static $hasTable = null;
    if ($hasTable === null) {
        $check = $conn->query("SHOW TABLES LIKE 'fleet_outages'");
        $hasTable = $check && $check->num_rows > 0;
        if ($check) { $check->free(); }
    }
    if (!$hasTable) return 0;

    $ts = $dt->format('Y-m-d H:i:s');
    $stmt = $conn->prepare("
      SELECT COUNT(DISTINCT vehicle_id) AS c
      FROM fleet_outages
      WHERE start_datetime <= ? AND end_datetime > ?
    ");
    $stmt->bind_param('ss', $ts, $ts);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res ? $res->fetch_assoc() : null;
    if ($res) { $res->free(); }
    $stmt->close();
    return $row ? (int)$row['c'] : 0;
}

// Final capacity for an exact datetime (override > global > physical), clamped by physical
function effective_capacity_for_dt(mysqli $conn, DateTime $dt): int {
    $active   = active_cars_count($conn);
    $down     = vehicles_down_count_at($conn, $dt);
    $physical = max(0, $active - $down);

    $ov = get_capacity_override_for_dt($conn, $dt);
    if ($ov !== null && $ov >= 0) return max(0, min($ov, $physical));

    $sv = get_setting($conn, 'active_fleet_capacity');
    if ($sv !== null && ctype_digit((string)$sv) && (int)$sv > 0) return max(0, min((int)$sv, $physical));

    return $physical;
}

// Count real drives (cars) running at a specific minute.
// Multiple students with the same instructor count as ONE car.
// Count cars (distinct slots) in use at a specific minute across the fleet
    // Count booked students at an exact minute (across ALL instructors)
// BookedDrives.slot_id -> InstructorAvailability(id) to get date/time
        function booked_students_at(mysqli $conn, DateTime $dt): int {
            $date = $dt->format('Y-m-d');
            $time = $dt->format('H:i:s');
            $stmt = $conn->prepare("
              SELECT COUNT(*) AS c
              FROM BookedDrives bd
              JOIN InstructorAvailability ia ON ia.id = bd.slot_id
              WHERE ia.available_date = ?
                AND ia.start_time    = ?
                AND ia.is_active     = 1
            ");
            $stmt->bind_param('ss', $date, $time);
            $stmt->execute();
            $row = $stmt->get_result()->fetch_assoc();
            return $row ? (int)$row['c'] : 0;
        }

            // Max students allowed in a single slot/car
            if (!defined('MAX_STUDENTS_PER_SLOT')) {
              define('MAX_STUDENTS_PER_SLOT', 3);
            }


/* ---------------------- POST: Admin booking ---------------------- */

if ($_SERVER["REQUEST_METHOD"] === "POST") {
    $student_id = (int)($_POST['student_id'] ?? 0);
    $slot_id    = (int)($_POST['slot_id'] ?? 0);

    if ($student_id && $slot_id) {
        // Load slot (and ensure not in the past)
    $stmt_slot = $conn->prepare("SELECT available_date, start_time, end_time, instructor_id FROM InstructorAvailability WHERE id=? AND is_active=1");
        $stmt_slot->bind_param("i", $slot_id);
        $stmt_slot->execute();
        $res_slot = $stmt_slot->get_result();
        $slot = $res_slot ? $res_slot->fetch_assoc() : null;
        if ($res_slot) { $res_slot->free(); }
        $stmt_slot->close();

        if (!$slot) {
            $error = "The selected slot could not be found.";
        } else {
            $slot_datetime = DateTime::createFromFormat('Y-m-d H:i:s', $slot['available_date'].' '.$slot['start_time'], $tz);
            $now = new DateTime('now', $tz);
            if (!$slot_datetime || $slot_datetime < $now) {
                $error = "This slot is already in the past and cannot be booked.";
            }
        }
        
        // Stop early if this student is already in that slot
            $dup = $conn->prepare("SELECT 1 FROM BookedDrives WHERE slot_id=? AND student_id=? LIMIT 1");
            $dup->bind_param("ii", $slot_id, $student_id);
            $dup->execute();
            if ($dup->get_result()->fetch_row()) {
                $error = "That student is already booked in that slot.";
            }


        if (!$error) {
            // Race-proof: advisory lock per minute
            $lockName = 'slot:'.$slot_id; // 1 lock per slot
            $lockStmt = $conn->prepare("SELECT GET_LOCK(?, 5)");
            $lockStmt->bind_param('s', $lockName);
            $lockStmt->execute();
            $lockRes = $lockStmt->get_result();
            $gotLockRow = $lockRes ? $lockRes->fetch_row() : [0];
            if ($lockRes) { $lockRes->free(); }
            $lockStmt->close();
            $gotLock = (int)($gotLockRow[0] ?? 0);

            if ($gotLock !== 1) {
                $error = "System is busy. Please try again.";
            } else {
                $txStarted = false;
                try {
                    
                    // How many are already in this slot?
                    $stmt = $conn->prepare("SELECT COUNT(*) AS c FROM BookedDrives WHERE slot_id=?");
                    $stmt->bind_param("i", $slot_id);
                    $stmt->execute();
                    $already = (int)$stmt->get_result()->fetch_assoc()['c'];
                    if ($already >= MAX_STUDENTS_PER_SLOT) {
                        throw new Exception("This slot is full.");
                    }

                    // 3b) enforce per-slot student capacity
                    $stmt = $conn->prepare("SELECT COUNT(*) FROM BookedDrives WHERE slot_id=?");
                    $stmt->bind_param("i", $slot_id);
                    $stmt->execute();
                    list($already) = $stmt->get_result()->fetch_row();
                    if ((int)$already >= MAX_STUDENTS_PER_SLOT) {
                        throw new Exception("This slot is full (".MAX_STUDENTS_PER_SLOT." students max).");
                    }

                    // cars available at this minute
                    $cap_cars = max(0, effective_capacity_for_dt($conn, $slot_datetime));
                    // students allowed across ALL cars at this minute
                    $cap_students = $cap_cars * MAX_STUDENTS_PER_SLOT;
                    
                    // how many students are already scheduled at this exact minute (all instructors)
                    $booked_students = booked_students_at($conn, $slot_datetime);

                    // Only block when we actually have a positive cap and it’s met/exceeded
                    if ($cap_students > 0 && $booked_students >= $cap_students) {
                        throw new Exception("Capacity reached for this time. Pick another slot.");
                    }

                    // Proceed to book
                    $conn->begin_transaction();
                    $txStarted = true;

                    $ins = $conn->prepare("INSERT INTO BookedDrives (student_id, slot_id) VALUES (?, ?)");
                    $ins->bind_param("ii", $student_id, $slot_id);
                    if (!$ins->execute()) {
                        $ins->close();
                        throw new Exception("Error booking drive.");
                    }
                    $ins->close();

                    // Mirror into DriveSchedule (if not already present)
                    $ins_ds = $conn->prepare("
                      INSERT INTO DriveSchedule
                        (drive_date, scheduled_time, instructor_id, student_id, minutes, created_at, lesson_status)
                      SELECT
                        ia.available_date,
                        ia.start_time,
                        ia.instructor_id,
                        ?,  /* student_id */
                        CASE
                          WHEN ia.start_time IS NULL OR ia.end_time IS NULL
                            THEN 60
                          ELSE TIME_TO_SEC(TIMEDIFF(ia.end_time, ia.start_time)) / 60
                        END AS minutes,
                        NOW(),
                        'P'
                      FROM InstructorAvailability ia
                      WHERE ia.id = ?
                        AND NOT EXISTS (
                          SELECT 1
                          FROM DriveSchedule ds
                          WHERE ds.instructor_id  = ia.instructor_id
                            AND ds.drive_date     = ia.available_date
                            AND ds.scheduled_time = ia.start_time
                            AND ds.student_id     = ?
                        )
                    ");
                    $ins_ds->bind_param('iii', $student_id, $slot_id, $student_id);
                    $ins_ds->execute();
                    $ins_ds->close();

                    // Fetch student & slot for email
                    $stmt_student = $conn->prepare("SELECT email, first_name FROM Students WHERE id=?");
                    $stmt_student->bind_param("i", $student_id);
                    $stmt_student->execute();
                    $res_stu = $stmt_student->get_result();
                    $student = $res_stu ? $res_stu->fetch_assoc() : null;
                    if ($res_stu) { $res_stu->free(); }
                    $stmt_student->close();

                    $stmt_slot = $conn->prepare("
                        SELECT ia.available_date, ia.start_time, ia.end_time, i.first_name, i.last_name
                        FROM InstructorAvailability ia
                        JOIN Instructors i ON ia.instructor_id = i.id
                        WHERE ia.id = ?
                    ");
                    $stmt_slot->bind_param("i", $slot_id);
                    $stmt_slot->execute();
                    $res_si = $stmt_slot->get_result();
                    $slotInfo = $res_si ? $res_si->fetch_assoc() : null;
                    if ($res_si) { $res_si->free(); }
                    $stmt_slot->close();

                    $conn->commit();

                    // Send confirmation email (best-effort)
                    if (!empty($student['email']) && $slotInfo) {
                        $subject = "Drive Time Booking Confirmation (Scheduled by Admin)";
                        $body_html = "
                            <p>Hello ".htmlspecialchars($student['first_name']).",</p>
                            <p>Your drive time has been scheduled by our team:</p>
                            <ul>
                                <li><strong>Date:</strong> ".htmlspecialchars($slotInfo['available_date'])."</li>
                                <li><strong>Time:</strong> ".htmlspecialchars($slotInfo['start_time'])." - ".htmlspecialchars($slotInfo['end_time'])."</li>
                                <li><strong>Instructor:</strong> ".htmlspecialchars($slotInfo['first_name'].' '.$slotInfo['last_name'])."</li>
                            </ul>
                            <p>If you have any questions, please contact our office or check your student dashboard.</p>
                            <hr>
                            <p style='font-size:0.9em;color:#888;'>This email is not monitored. Please forward all requests to <a href='mailto:info@tandddrivingacademy.com'>info@tandddrivingacademy.com</a>.</p>
                        ";
                        // ignore result; don't block UI on email failures
                        send_email($student['email'], $student['first_name'], $subject, $body_html);
                    }

                    $_SESSION['booking_success'] = "Drive scheduled successfully.";
                    header("Location: admin-drive-booking.php?success=1");
                    exit();

                } catch (mysqli_sql_exception $ex) {
                    if ($txStarted) { $conn->rollback(); }
                    if ((int)$ex->getCode() === 1062) { // duplicate key
                        // unique key on (student_id, slot_id)
                        $error = "That student is already booked in that slot.";
                    } else {
                        $error = "Database error: " . $ex->getMessage();
                    }
                } catch (Exception $ex) {
                    if ($txStarted) { $conn->rollback(); }
                    $error = $ex->getMessage();
                } finally {
                    $rel = $conn->prepare("SELECT RELEASE_LOCK(?)");
                    $rel->bind_param('s', $lockName);
                    $rel->execute();
                }
            }
        }
    } else {
        $error = "Please select a student and time slot.";
    }
}

// Flash success on redirect
if (isset($_GET['success']) && isset($_SESSION['booking_success'])) {
    $success = $_SESSION['booking_success'];
    unset($_SESSION['booking_success']);
}

/* ---------------------- Data for form ---------------------- */

$students = $conn->query("SELECT id, first_name, last_name FROM Students ORDER BY last_name, first_name");

$available_slots = $conn->query("
    SELECT
      ia.id,
      ia.available_date,
      ia.start_time,
      ia.end_time,
      i.first_name,
      i.last_name,
      COUNT(bd.id) AS booked_count
    FROM InstructorAvailability ia
    JOIN Instructors i ON i.id = ia.instructor_id
    LEFT JOIN BookedDrives bd ON bd.slot_id = ia.id
    WHERE ia.is_active = 1
      AND CONCAT(ia.available_date, ' ', ia.start_time) >= NOW()
    GROUP BY ia.id, ia.available_date, ia.start_time, ia.end_time, i.first_name, i.last_name
    HAVING booked_count < " . MAX_STUDENTS_PER_SLOT . "
    ORDER BY ia.available_date, ia.start_time
");

?>
<!DOCTYPE html>
<html lang="en">
<head>
  <link rel="stylesheet" href="styles.css">
  <meta charset="UTF-8">
  <title>Admin Drive Booking</title>
  <style>
    body { font-family: Arial, sans-serif; padding: 20px; }
    .container { max-width: 600px; margin: auto; background: rgba(255,255,255,0.75); padding: 25px; border-radius: 10px; box-shadow: 0 0 10px rgba(0,0,0,0.1); transform: translateY(260px);}
    h2 { text-align: center; }
    label, select, input[type=submit], .dashboard-btn { display: block; width: 100%; margin-top: 10px; padding: 10px; }
    input[type=submit], .dashboard-btn { background: #007bff; color: white; border: none; cursor: pointer; border-radius: 5px; }
    .search-wrapper { position: relative; width: 100%; }
    #suggestions { position: absolute; top: 100%; left: 0; width: 100%; z-index: 1000; background: white; border: 1px solid #ccc; border-top: none; max-height: 200px; overflow-y: auto; box-shadow: 0 2px 5px rgba(0,0,0,0.2); }
    .success { color: green; text-align: center; }
    .error { color: red; text-align: center; }
  </style>
</head>
<body>
<div class="container">
  <h2>Schedule a Drive Time</h2>

  <?php if ($success): ?><p class="success"><?= htmlspecialchars($success) ?></p><?php endif; ?>
  <?php if ($error): ?><p class="error"><?= htmlspecialchars($error) ?></p><?php endif; ?>

  <form method="post">
    <label for="student_id">Select Student:</label>
    <label for="student-search">Search Student:</label>

    <div class="search-wrapper">
      <input type="text" id="student-search" placeholder="Start typing..." autocomplete="off" required>
      <div id="suggestions"></div>
    </div>

    <input type="hidden" name="student_id" id="student_id" required>

    <label for="slot_id">Select Available Time Slot:</label>
    <select name="slot_id" id="slot_id" required>
      <option value="">-- Select Slot --</option>
      <?php while ($slot = $available_slots->fetch_assoc()): ?>
          <option value="<?= (int)$slot['id'] ?>">
            <?= htmlspecialchars($slot['available_date']) ?> |
            <?php
              $startFormatted = date("g:i A", strtotime($slot['start_time']));
              $endFormatted   = date("g:i A", strtotime($slot['end_time']));
              echo htmlspecialchars($startFormatted . ' - ' . $endFormatted);
            ?>
            with <?= htmlspecialchars($slot['first_name'] . ' ' . $slot['last_name']) ?>
            (<?= (int)$slot['booked_count'] ?>/<?= MAX_STUDENTS_PER_SLOT ?>)
          </option>
        <?php endwhile; ?>
    </select>

    <input type="submit" value="Book Drive">
  </form>

  <form method="get" action="admin-dashboard.php">
    <button type="submit" class="dashboard-btn">Return to Dashboard</button>
  </form>
</div>
<script src="js/admin-drive-search-engine.js"></script>
</body>
</html>




