<?php
session_start();
require 'db.php';
$conn->set_charset('utf8mb4');
require_once 'includes/send_email.php';

if (function_exists('mysqli_report')) {
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
}

if (!isset($_SESSION['admin_id'])) {
    header("Location: admin-login.php");
    exit();
}

$success = "";
$error = "";
$current_date = date("Y-m-d");

// ---------------------------
// POST: Put chosen instructor on this CAR/TIME
// - Only if instructor is FREE at that date/time
// - If busy (same day & overlapping window), show error; DO NOT swap
// ---------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['slot_id'], $_POST['new_instructor_id'])) {
    $slot_id = (int)$_POST['slot_id'];
    $new_instructor_id = (int)$_POST['new_instructor_id'];

    try {
        // A) Load this slot (date/time + current instructor)
        $stmt = $conn->prepare("
            SELECT car_id, instructor_id, available_date, start_time, end_time
            FROM InstructorAvailability
            WHERE id = ?
            LIMIT 1
        ");
        $stmt->bind_param("i", $slot_id);
        $stmt->execute();
        $stmt->bind_result($car_id_a, $inst_id_a, $d, $t_start, $t_end);
        $found = $stmt->fetch();
        $stmt->close();

        if (!$found) {
            throw new RuntimeException("Slot not found.");
        }
        if ($new_instructor_id === $inst_id_a) {
            $success = "No change: that instructor already has this car/time.";
        } else {
            // B) Is the target instructor already busy at this DATE with an OVERLAPPING time?
            // overlap test: (existing.start < this.end) AND (existing.end > this.start)
            $stmt = $conn->prepare("
                SELECT id
                FROM InstructorAvailability
                WHERE instructor_id = ?
                  AND available_date = ?
                  AND start_time < ?
                  AND end_time   > ?
                  AND id <> ?
                LIMIT 1
            ");
            $stmt->bind_param("isssi", $new_instructor_id, $d, $t_end, $t_start, $slot_id);
            $stmt->execute();
            $stmt->bind_result($conflict_slot_id);
            $hasConflict = $stmt->fetch();
            $stmt->close();

            if ($hasConflict) {
                // Hard stop: they’re already scheduled at this time (maybe on another car).
                $pretty_start = date("g:i A", strtotime($t_start));
                $pretty_end   = date("g:i A", strtotime($t_end));
                $error = "That instructor is already scheduled on {$d} {$pretty_start}–{$pretty_end}. Choose a different instructor or reschedule to a different time/date.";
            } else {
                // C) Instructor is free → simple reassignment
                $stmt_upd = $conn->prepare("UPDATE InstructorAvailability SET instructor_id = ? WHERE id = ?");
                $stmt_upd->bind_param("ii", $new_instructor_id, $slot_id);
                $stmt_upd->execute();
                $stmt_upd->close();

                // Refresh (PRG) so the table updates and avoids resubmission
                header('Location: ' . $_SERVER['PHP_SELF'] . '?t=' . time());
                exit;
            }
        }
    } catch (Throwable $e) {
        error_log("Car/instructor assignment failed: " . $e->getMessage());
        $error = "Car/instructor assignment failed: " . $e->getMessage();
    }
}
// ---------------------------
// LIST: Upcoming car slots with current instructor (+ booked count)
// ---------------------------
$slots = [];
try {
    $capacity_default = 3; // no schema changes; hard-code capacity

    $sql = "
      SELECT ia.id AS slot_id,
             ia.available_date, ia.start_time, ia.end_time,
             c.id AS car_id, c.name AS car_name,
             i.id AS instructor_id, i.first_name, i.last_name,
             COALESCE(bd_cnt.cnt, 0) AS booked_count
      FROM InstructorAvailability ia
      JOIN Cars c        ON c.id = ia.car_id
      JOIN Instructors i ON i.id = ia.instructor_id
      LEFT JOIN (
          SELECT slot_id, COUNT(*) AS cnt
          FROM BookedDrives
          GROUP BY slot_id
      ) bd_cnt ON bd_cnt.slot_id = ia.id
      WHERE ia.available_date >= ? AND ia.is_active = 1
        ORDER BY ia.available_date, ia.start_time, c.name
    ";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param('s', $current_date);
    $stmt->execute();

    // Bind EXACTLY the columns selected above (10 total)
    $stmt->bind_result(
        $slot_id, $d, $t_start, $t_end,
        $car_id, $car_name,
        $inst_id, $inst_first, $inst_last,
        $booked_count
    );

    while ($stmt->fetch()) {
        $slots[] = [
            'slot_id'        => $slot_id,
            'available_date' => $d,
            'start_time'     => $t_start,
            'end_time'       => $t_end,
            'car_id'         => $car_id,
            'car_name'       => $car_name,
            'instructor_id'  => $inst_id,
            'inst_first'     => $inst_first,
            'inst_last'      => $inst_last,
            'booked_count'   => (int)$booked_count,
            'capacity'       => $capacity_default, // <-- supply capacity here
        ];
    }
    $stmt->close();
} catch (Throwable $e) {
    error_log('Slots query failed: ' . $e->getMessage());
    $error = "Database query failed (slots): " . $e->getMessage();
}

// --- Build instructors list (robust, no prepare) ---
$instructor_options = [];
$sql = "SELECT `id`, `first_name`, `last_name` FROM `Instructors` ORDER BY `last_name`, `first_name`";
$res = $conn->query($sql);

if ($res instanceof mysqli_result) {
    while ($row = $res->fetch_assoc()) {
        $instructor_options[] = [
            'id'    => (int)$row['id'],
            'first' => $row['first_name'],
            'last'  => $row['last_name'],
        ];
    }
    $res->free();
} else {
    // Surface the real reason so you’re not stuck with a blank page
    $error = "Instructors query failed: (" . $conn->errno . ") " . $conn->error;
    error_log($error . " | SQL=" . $sql);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Admin: Reassign Instructors by Car</title>
  <link rel="stylesheet" href="styles.css">
  <style>
    body { font-family: Arial, sans-serif; padding: 20px; }
    table { width: 100%; border-collapse: collapse; margin-top: 20px; }
    th, td { padding: 10px; border: 1px solid #ccc; text-align: left; }
    th { background: #eee; }
    form.inline { display: inline; }
    select, button { padding: 6px; border-radius: 4px; }
    .nav-btn { margin-top: 20px; padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; text-decoration: none; }
    .pill { display:inline-block; padding:2px 8px; border-radius:12px; background:#f0f0f0; font-size:12px; }
    .ok { color:#0a0; } .warn { color:#b00; }
    .table-container {
      background: rgba(255, 255, 255, 0.85);
      padding: 25px;
      border-radius: 10px;
      box-shadow: 0 0 12px rgba(0, 0, 0, 0.2);
      margin-top: 30px;
    }
  </style>
</head>
<body>
  <h2 style="color: white;">Admin: Reassign Instructors by Car</h2>

  <?php if ($success): ?>
    <p style="color: white; text-align: center; font-weight: bold; text-shadow: 1px 1px 3px rgba(0,0,0,0.7);">
      <?= htmlspecialchars($success) ?>
    </p>
  <?php endif; ?>

  <?php if ($error): ?>
    <p style="color: #d33; text-align: center; font-weight: bold;">
      <?= htmlspecialchars($error) ?>
    </p>
  <?php endif; ?>

  <div class="table-container">
    <table>
      <thead>
        <tr>
          <th>Car</th>
          <th>Date</th>
          <th>Time</th>
          <th>Current Instructor</th>
          <th>Booked</th>
          <th>New Instructor</th>
          <th>Action</th>
        </tr>
      </thead>
      <tbody>
      <?php if (!empty($slots)): ?>
        <?php foreach ($slots as $slot): ?>
          <tr>
            <td><?= htmlspecialchars($slot['car_name']) ?></td>
            <td><?= htmlspecialchars($slot['available_date']) ?></td>
            <td><?= date("g:i A", strtotime($slot['start_time'])) ?> - <?= date("g:i A", strtotime($slot['end_time'])) ?></td>
            <td><?= htmlspecialchars($slot['inst_first'] . ' ' . $slot['inst_last']) ?></td>
            <td>
              <span class="pill <?= ($slot['booked_count'] >= $slot['capacity']) ? 'warn' : 'ok' ?>">
                <?= (int)$slot['booked_count'] ?> / <?= (int)$slot['capacity'] ?>
              </span>
            </td>
            <td>
              <form method="post" class="inline">
                <input type="hidden" name="slot_id" value="<?= (int)$slot['slot_id'] ?>">
                <select name="new_instructor_id" required>
                  <option value="">-- Select --</option>
                  <?php foreach ($instructor_options as $inst): ?>
                    <option value="<?= (int)$inst['id'] ?>">
                      <?= htmlspecialchars($inst['last'] . ', ' . $inst['first']) ?>
                    </option>
                  <?php endforeach; ?>
                </select>
            </td>
            <td>
                <button type="submit">Reassign</button>
              </form>
            </td>
          </tr>
        <?php endforeach; ?>
      <?php else: ?>
        <tr><td colspan="7">No upcoming car slots found.</td></tr>
      <?php endif; ?>
      </tbody>
    </table>
  </div>

  <div style="text-align:center;">
    <a href="admin-dashboard.php" class="nav-btn">Back to Dashboard</a>
  </div>
</body>
</html>


