<?php
// admin-log-drive.php
session_start();
require 'db.php';
// require 'includes/auth_admin.php'; // <-- adjust if your admin guard lives elsewhere

if (!function_exists('column_exists')) {
  function column_exists(mysqli $conn, string $table, string $col): bool {
    $sql = "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('ss', $table, $col);
    $stmt->execute();
    $stmt->bind_result($cnt);
    $stmt->fetch();
    $stmt->close();
    return ($cnt > 0);
  }
}

require 'includes/PHPMailer/PHPMailer.php';
require 'includes/PHPMailer/SMTP.php';
require 'includes/PHPMailer/Exception.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

$success = "";
$error = "";

// ----- helpers -----
function safe($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

// ----- fetch lesson groups and instructors for the form -----
$lesson_groups = [];
$resLG = $conn->query("SELECT id, code, label FROM lesson_groups ORDER BY id ASC");
while ($row = $resLG->fetch_assoc()) {
  $lesson_groups[$row['code']] = ['id' => (int)$row['id'], 'label' => $row['label']];
}
if ($resLG) $resLG->free();

$instructors = [];
$resI = $conn->query("SELECT id, first_name, last_name FROM Instructors ORDER BY last_name, first_name");
while ($r = $resI->fetch_assoc()) {
  $instructors[] = [
    'id' => (int)$r['id'],
    'name' => trim(($r['last_name'] ?? '').', '.($r['first_name'] ?? '')) ?: ('#'.(int)$r['id']),
  ];
}
if ($resI) $resI->free();

// ----- POST: log drive -----
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $student_id     = isset($_POST['student_id']) ? (int)$_POST['student_id'] : 0;
  $btw_minutes    = isset($_POST['btw_minutes']) ? max(0, (int)$_POST['btw_minutes']) : 0;
  $ob_minutes     = isset($_POST['ob_minutes']) ? max(0, (int)$_POST['ob_minutes']) : 0;
  $minutes        = $btw_minutes; // overall progress mirrors BTW
  $drive_date     = $_POST['date']            ?? '';
  $scheduled_time = $_POST['scheduled_time'] ?? '';
    if ($scheduled_time && strlen($scheduled_time) === 5) { // e.g. "14:30"
      $scheduled_time .= ':00';
    }
  $lesson_group   = isset($_POST['lesson_group']) ? trim($_POST['lesson_group']) : '';
  $lesson_status  = isset($_POST['lesson_status']) ? trim($_POST['lesson_status']) : 'P';
  $chosen_instructor_id = isset($_POST['instructor_id']) ? (int)$_POST['instructor_id'] : 0;

  $entry_type      = $_POST['entry_type'] ?? 'completed'; // 'completed' | 'no_show'
  $no_show_reason  = trim($_POST['no_show_reason'] ?? '');
  $no_show_fee_in  = trim($_POST['no_show_fee'] ?? '');
  $recorded_by     = 'admin';

  /* =========================
     NO-SHOW BRANCH (requires a booked slot)
     ========================= */
  if ($entry_type === 'no_show') {
    // find the matching booking and its instructor
    $slotQ = $conn->prepare("
      SELECT bd.id AS booked_id, ia.instructor_id
      FROM BookedDrives bd
      JOIN InstructorAvailability ia ON ia.id = bd.slot_id
      WHERE bd.student_id = ? AND ia.available_date = ? AND ia.start_time = ?
      LIMIT 1
    ");
    $slotQ->bind_param('iss', $student_id, $drive_date, $scheduled_time);
    $slotQ->execute();
    $slot = $slotQ->get_result()->fetch_assoc();
    $slotQ->close();

    if (!$slot) {
      $error = "No matching booked slot for the selected student/date/time.";
    } else {
      $slot_id            = (int)$slot['booked_id'];
      $slot_instructor_id = (int)$slot['instructor_id'];

      // prevent duplicate log (mirror unique key: student_id + drive_date + scheduled_time)
        $dupe = $conn->prepare("
          SELECT 1 FROM DriveSchedule
          WHERE student_id=? AND drive_date=? AND scheduled_time=?
          LIMIT 1
        ");
        $dupe->bind_param('iss', $student_id, $drive_date, $scheduled_time);
        $dupe->execute();
        $exists = (bool)$dupe->get_result()->fetch_row();
        $dupe->close();
        
        if ($exists) {
          $error = "This slot was already logged.";
        } else {
        // DriveSchedule row with status 'N'
        $ins = $conn->prepare("
          INSERT INTO DriveSchedule
            (student_id, instructor_id, scheduled_time, minutes, drive_date,
             lesson_group_id, lesson_status, btw_minutes, ob_minutes)
          VALUES (?, ?, ?, 0, ?, NULL, 'N', 0, 0)
        ");
        $ins->bind_param('iiss', $student_id, $slot_instructor_id, $scheduled_time, $drive_date);
        if ($ins->execute()) {
          $success = "No-show recorded for {$drive_date} at {$scheduled_time}.";
        } else {
          $error = "Failed to record no-show (DriveSchedule).";
        }
        $ins->close();

        // Optional: flag booking if that column exists
        if (empty($error) && function_exists('column_exists') && column_exists($conn, 'BookedDrives', 'status')) {
          $upd = $conn->prepare("UPDATE BookedDrives SET status='no_show' WHERE id=? LIMIT 1");
          $upd->bind_param('i', $slot_id);
          $upd->execute();
          $upd->close();
        }

        // Insert into NoShows (fee nullable)
        if (empty($error)) {
          $fee = ($no_show_fee_in === '' ? null : round((float)$no_show_fee_in, 2));
          if ($fee === null) {
            $ns = $conn->prepare("INSERT INTO `NoShows` (student_id, date, reason, recorded_by) VALUES (?,?,?,?)");
            $ns->bind_param('isss', $student_id, $drive_date, $no_show_reason, $recorded_by);
          } else {
            $ns = $conn->prepare("INSERT INTO `NoShows` (student_id, date, fee, reason, recorded_by) VALUES (?,?,?,?,?)");
            $ns->bind_param('isdss', $student_id, $drive_date, $fee, $no_show_reason, $recorded_by);
          }
          if ($ns) { $ns->execute(); $ns->close(); }
          // Email student about no-show (same SMTP config you already use)
        $es = $conn->prepare("SELECT email, first_name, last_name FROM Students WHERE id = ?");
        $es->bind_param('i', $student_id);
        $es->execute();
        if ($stu = $es->get_result()->fetch_assoc()) {
            $mail = new PHPMailer(true);
            try {
                $mail->isSMTP();
                $mail->Host = 'mail.tandddrivingacademy.com';
                $mail->SMTPAuth = true;
                $mail->Username = 'info@tandddrivingacademy.com';
                $mail->Password = '!DriverightTX!';
                $mail->SMTPSecure = 'ssl';
                $mail->Port = 465;
        
                $mail->setFrom('info@tandddrivingacademy.com', 'T&D Driving Academy');
                $mail->addAddress($stu['email'], $stu['first_name'].' '.$stu['last_name']);
                $mail->isHTML(true);
                $mail->Subject = 'No-Show Recorded for Scheduled Drive';
        
                // recompute fee for this scope
                $feeVal   = ($no_show_fee_in === '' ? null : round((float)$no_show_fee_in, 2));
                $feeLine  = $feeVal !== null ? "<br><strong>Fee:</strong> $".number_format($feeVal, 2) : "";
                $reason   = $no_show_reason !== '' ? htmlspecialchars($no_show_reason, ENT_QUOTES, 'UTF-8') : '';
                $reasonLine = $reason !== '' ? "<br><strong>Reason:</strong> {$reason}" : '';
        
                $mail->Body = "Dear {$stu['first_name']} {$stu['last_name']},<br><br>
                    Our records show you were marked as a <strong>no-show</strong> for your scheduled drive on
                    <strong>{$drive_date}</strong> at <strong>{$scheduled_time}</strong>.{$reasonLine}{$feeLine}
                    <br><br>If you need to cancel a booking, you need to do so via email before noon of the day of your booked driving lesson.<br><br>
                    Thank you,<br>T&amp;D Driving Academy";
        
                $mail->send();
            } catch (Exception $e) {
                error_log('No-show email failed: '.$mail->ErrorInfo);
            }
        }
        $es->close();
        }
      }
    }
    // done with no-show branch
  }
  /* =========================
     COMPLETED BRANCH
     ========================= */
  else {
    // validations only for completed logs
    $valid_statuses = ['I','F','P'];
    if (!$student_id)        { $error = "Please select a student."; }
    elseif (!$drive_date)    { $error = "Please provide a drive date."; }
    elseif (!$scheduled_time){ $error = "Please provide a scheduled time."; }
    elseif (!in_array($lesson_status, $valid_statuses, true)) { $error = "Invalid lesson status."; }
    elseif (empty($lesson_group) || !isset($lesson_groups[$lesson_group])) { $error = "Invalid lesson group."; }
    elseif ($btw_minutes + $ob_minutes <= 0) { $error = "Please enter some minutes to log."; }

    if (!$error) {
      $lesson_group_id = (int)$lesson_groups[$lesson_group]['id'];
      // daily caps
      $chk = $conn->prepare("
        SELECT COALESCE(SUM(btw_minutes),0), COALESCE(SUM(minutes),0)
        FROM DriveSchedule
        WHERE student_id=? AND drive_date=?
      ");
      $chk->bind_param('is', $student_id, $drive_date);
      $chk->execute();
      $chk->bind_result($sumBTW, $sumAny);
      $chk->fetch(); $chk->close();

      if ($sumBTW + $btw_minutes > 120)      { $error = "Student would exceed the 2-hour BTW limit for this day."; }
      elseif ($sumAny + $btw_minutes + $ob_minutes > 300) { $error = "Student would exceed the 5-hour total limit for this day."; }
    }

    // resolve instructor: prefer booked, else dropdown
    $instructor_id = 0;
    if (!$error) {
      $own = $conn->prepare("
        SELECT ia.instructor_id
        FROM BookedDrives bd
        JOIN InstructorAvailability ia ON ia.id = bd.slot_id
        WHERE bd.student_id=? AND ia.available_date=? AND ia.start_time=? LIMIT 1
      ");
      $own->bind_param('iss', $student_id, $drive_date, $scheduled_time);
      $own->execute();
      $own->bind_result($resolved_instructor_id);
      if ($own->fetch()) $instructor_id = (int)$resolved_instructor_id;
      $own->close();

      if ($instructor_id === 0) {
        if ($chosen_instructor_id > 0) $instructor_id = $chosen_instructor_id;
        else $error = "No booking found for this date/time. Please select an instructor.";
      }
    }

    // block duplicate
    if (!$error) {
      $du = $conn->prepare("SELECT 1 FROM DriveSchedule WHERE student_id=? AND drive_date=? AND scheduled_time=? LIMIT 1");
      $du->bind_param('iss', $student_id, $drive_date, $scheduled_time);
      $du->execute(); $du->store_result();
      if ($du->num_rows > 0) $error = "A drive at this date/time is already logged for this student.";
      $du->close();
    }

    // disallow future logging
    if (!$error) {
      $now = new DateTime('now', new DateTimeZone('America/Chicago'));
      $slotDT = DateTime::createFromFormat('Y-m-d H:i:s', $drive_date.' '.$scheduled_time, new DateTimeZone('America/Chicago'));
      if ($slotDT && $slotDT > $now) { $error = "You can’t log a future drive."; }
    }

    // insert + email
    if (!$error) {
      $stmt = $conn->prepare("
        INSERT INTO DriveSchedule
          (student_id, instructor_id, scheduled_time, minutes, drive_date, lesson_group_id, lesson_status, btw_minutes, ob_minutes)
        VALUES (?,?,?,?,?,?,?,?,?)
      ");
      $stmt->bind_param('iisisisii', $student_id, $instructor_id, $scheduled_time, $minutes, $drive_date, $lesson_group_id, $lesson_status, $btw_minutes, $ob_minutes);
      if ($stmt->execute()) {
        $success = "Drive time logged successfully.";
        // notify student (same config you already use)
        $es = $conn->prepare("SELECT email, first_name, last_name FROM Students WHERE id = ?");
        $es->bind_param('i', $student_id);
        $es->execute();
        if ($stu = $es->get_result()->fetch_assoc()) {
          $mail = new PHPMailer(true);
          try {
            $mail->isSMTP();
            $mail->Host = 'mail.tandddrivingacademy.com';
            $mail->SMTPAuth = true;
            $mail->Username = 'info@tandddrivingacademy.com';
            $mail->Password = '!DriverightTX!';
            $mail->SMTPSecure = 'ssl';
            $mail->Port = 465;

            $mail->setFrom('info@tandddrivingacademy.com', 'T&D Driving Academy');
            $mail->addAddress($stu['email'], $stu['first_name'].' '.$stu['last_name']);
            $mail->isHTML(true);
            $mail->Subject = 'Drive Time Logged';
            $mail->Body = "Dear {$stu['first_name']} {$stu['last_name']},<br><br>
              Your drive time of {$minutes} minutes (BTW {$btw_minutes}, OB {$ob_minutes}) on {$drive_date} at {$scheduled_time} has been logged.<br><br>
              Thank you,<br>T&D Driving Academy";
            $mail->send();
          } catch (Exception $e) {
            error_log("Email could not be sent. Mailer Error: {$mail->ErrorInfo}");
          }
        }
        $es->close();
      } else {
        $error = "Failed to log drive.";
      }
      $stmt->close();
    }
  } // end completed
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <title>Admin • Log Drive Time</title>
  <style>
    :root{
      --bg1:#0a1930; --bg2:#0f2e5e; --bg3:#1a437a;
      --text:#f5f9ff; --muted:#a9c1e6; --card-bg:rgba(255,255,255,.06);
      --card-border:rgba(255,255,255,.14); --accent:#58b1ff; --danger:#ff6b7a;
      --input-bg:#0f2142; --input-border:rgba(255,255,255,.18);
    }
    *{box-sizing:border-box}
    body{
      font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,sans-serif;
      margin:24px; color:var(--text);
      background:
        radial-gradient(1200px 800px at 15% -10%, var(--bg3) 0%, transparent 60%),
        radial-gradient(1000px 700px at 110% 10%, var(--bg2) 0%, transparent 55%),
        linear-gradient(180deg, var(--bg1) 0%, #081223 100%);
    }
    h1,h2{margin:0 0 12px}
    .row{display:grid;grid-template-columns:1fr 1fr;gap:16px}
    .card{background:var(--card-bg);border:1px solid var(--card-border);border-radius:14px;padding:16px;margin:16px 0}
    label{display:block;margin:.5rem 0 .25rem;color:var(--muted);font-weight:600}
    input,select{
      width:100%;padding:.6rem .75rem;border-radius:10px;border:1px solid var(--input-border);
      background:var(--input-bg);color:var(--text)
    }
    .btn{display:inline-block;background:linear-gradient(180deg, var(--accent) 0%, #3da6ff 100%);color:#00132a;border:0;border-radius:10px;padding:.65rem 1.05rem;font-weight:700;cursor:pointer}
    .btn-secondary{background:#23385f;color:var(--text)}
    .btn-danger{background:linear-gradient(180deg, var(--danger) 0%, #ff4960 100%);color:#2a0006}
    table{width:100%;border-collapse:collapse;margin-top:12px}
    th,td{border:1px solid var(--card-border);padding:8px;text-align:left}
    .ok{color:#86ffd0}.err{color:#ff9aa8}
    #suggestions{position:relative;margin-top:4px}
  </style>
</head>
<body>
    <div style="display:flex; justify-content:flex-end; margin-bottom:12px;">
  <a href="https://bookings.tandddrivingacademy.com/admin-dashboard.php"
     class="btn"
     style="text-decoration:none;">
    ← Admin Dashboard
  </a>
</div>
  <h2>Admin • Log Student Drive</h2>
  <?php if ($success): ?>
    <div class="card ok"><strong><?php echo safe($success); ?></strong></div>
    <div class="row">
      <form method="get"><button class="btn">Log Another</button></form>
      <a href="admin-dashboard.php" class="btn btn-secondary" style="text-decoration:none;display:inline-flex;align-items:center;justify-content:center">← Back to Dashboard</a>
    </div>
  <?php elseif ($error): ?>
    <div class="card err"><strong><?php echo safe($error); ?></strong></div>
  <?php endif; ?>

  <?php if (!$success): ?>
  <div class="card">
    <form method="post" id="logDriveForm">
      <div class="row">
        <div>
          <label for="student_search">Student</label>
          <input type="text" id="student_search" placeholder="Type student name…" autocomplete="off">
          <input type="hidden" name="student_id" id="student_id">
          <div id="suggestions"></div>
        </div>
        <div>
          <label for="instructor_id">Instructor (if no booking match)</label>
          <select name="instructor_id" id="instructor_id">
            <option value="0">— Select instructor —</option>
            <?php foreach ($instructors as $ins): ?>
              <option value="<?php echo (int)$ins['id']; ?>"><?php echo safe($ins['name']); ?></option>
            <?php endforeach; ?>
          </select>
          <div style="color:#a9c1e6;font-size:.9rem;margin-top:.25rem;">
            If a booking exists for this date/time, that instructor will be used automatically.
          </div>
        </div>
      </div>
      <label style="font-weight:700">Entry Type</label>
        <div style="display:flex;gap:16px;align-items:center;margin:.25rem 0 .5rem">
          <label><input type="radio" name="entry_type" value="completed" checked> Completed</label>
          <label><input type="radio" name="entry_type" value="no_show"> No-Show</label>
        </div>
        <div id="completedFields">
      <div class="row">
        <div>
          <label for="btw_minutes">Behind-the-Wheel Minutes</label>
          <input type="number" name="btw_minutes" id="btw_minutes" min="0" max="120" step="5" placeholder="e.g., 40">
        </div>
        <div>
          <label for="ob_minutes">Observation Minutes</label>
          <input type="number" name="ob_minutes" id="ob_minutes" min="0" max="120" step="5" placeholder="e.g., 80">
        </div>
      </div>

      <div class="row">
        <div>
          <label for="lesson_group">Lesson Group Covered</label>
          <select name="lesson_group" id="lesson_group" required>
            <option value="">— Select a Lesson Group —</option>
            <?php foreach ($lesson_groups as $code => $meta): ?>
              <option value="<?php echo safe($code); ?>"><?php echo safe($meta['label']); ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label for="lesson_status">Result</label>
          <select name="lesson_status" id="lesson_status" required>
            <option value="P">Pass</option>
            <option value="F">Fail</option>
            <option value="I">Incomplete</option>
          </select>
        </div>
      </div>
    </div>
      <div class="row">
        <div>
          <label for="date">Drive Date</label>
          <input type="date" name="date" id="date" required>
        </div>
        <div>
          <label for="scheduled_time">Scheduled Time</label>
          <input type="time" name="scheduled_time" id="scheduled_time" required>
        </div>
      </div>
        <div id="noShowFields" style="display:none">
      <label for="no_show_reason">No-Show Reason (optional)</label>
      <input type="text" name="no_show_reason" id="no_show_reason" placeholder="e.g., student absent, weather, vehicle down">
    
      <label for="no_show_fee">No-Show Fee (optional)</label>
      <input type="number" name="no_show_fee" id="no_show_fee" step="0.01" min="0" placeholder="e.g., 35.00">
    </div>
      <div style="margin-top:12px">
        <button class="btn">Log Drive Time</button>
        <a href="admin-dashboard.php" class="btn btn-secondary" style="text-decoration:none;margin-left:8px">Cancel</a>
      </div>
    </form>
  </div>

  <div class="card">
    <h3 style="margin-top:0">Lesson History</h3>
    <p style="margin:0 0 8px;color:#a9c1e6">Select a student to view recent logs.</p>
    <div id="lesson-history">
      <!-- You can populate via JS when student is selected -->
      <!-- Or server-render by posting student_id to this page and querying DriveSchedule -->
    </div>
  </div>
  <?php endif; ?>

  <script src="js/log-drive-search.js"></script>
 <script>
  // Export lesson groups (code => label) for any JS that needs it
  const LESSON_GROUPS = <?php
    $labels = [];
    foreach ($lesson_groups as $code => $meta) { $labels[$code] = $meta['label']; }
    echo json_encode($labels, JSON_UNESCAPED_UNICODE);
  ?>;

  (function () {
    const radios    = document.querySelectorAll('input[name="entry_type"]');
    const completed = document.getElementById('completedFields');
    const noShow    = document.getElementById('noShowFields');
    const btw       = document.getElementById('btw_minutes');
    const ob        = document.getElementById('ob_minutes');
    const lg        = document.getElementById('lesson_group');
    const ls        = document.getElementById('lesson_status');

    function refresh() {
      const val = document.querySelector('input[name="entry_type"]:checked')?.value || 'completed';
      const isNoShow = (val === 'no_show');

      if (completed) completed.style.display = isNoShow ? 'none' : '';
      if (noShow)    noShow.style.display    = isNoShow ? '' : 'none';

      // Toggle required attrs so the browser doesn’t block submit on No-Show
      if (lg) { isNoShow ? lg.removeAttribute('required') : lg.setAttribute('required', 'required'); }
      if (ls) { isNoShow ? ls.removeAttribute('required') : ls.setAttribute('required', 'required'); }

      // Clear minute fields when switching to No-Show
      if (isNoShow) { if (btw) btw.value=''; if (ob) ob.value=''; }
    }

    radios.forEach(r => r.addEventListener('change', refresh));
    refresh();

    // Clamp minutes on input (BTW ≤ 120, OB ≤ 120)
    function clampOne(el, max) {
      if (!el) return;
      const v = parseInt(el.value || 0, 10);
      if (Number.isFinite(v)) el.value = Math.max(0, Math.min(max, v));
    }
    function clamp(){ clampOne(btw,120); clampOne(ob,120); }
    [btw, ob].forEach(el => el && el.addEventListener('input', clamp));
  })();
</script>
</body>
</html>

