<?php
// admin-master-scheduler.php
require_once 'db.php';
// require_once 'includes/auth_admin.php'; // adjust if you use a different admin guard

$tz = new DateTimeZone('America/Chicago');

// ---------- helpers ----------
function dt($date, $time, $tz) {
  return DateTime::createFromFormat('Y-m-d H:i', "$date $time", $tz);
}
function fmt($dt) { return $dt->format('Y-m-d H:i:s'); }

// Active fleet count (status='active')
function get_active_fleet_count($conn) {
  $rs = $conn->query("SELECT COUNT(*) AS c FROM fleet WHERE status='active'");
  $row = $rs ? $rs->fetch_assoc() : null;
  return $row ? (int)$row['c'] : 0;
}

// Concurrent booked drives at a timestamp
function concurrent_drives_at($conn, $when) {
  $date = $when->format('Y-m-d');
  $time = $when->format('H:i:s');
  $stmt = $conn->prepare("SELECT COUNT(*) AS c FROM DriveSchedule WHERE drive_date=? AND scheduled_time=?");
  $stmt->bind_param('ss', $date, $time);
  $stmt->execute();
  $row = $stmt->get_result()->fetch_assoc();
  return $row ? (int)$row['c'] : 0;
}

// Overlap with existing availability (same instructor)
function availability_overlap_exists($conn, $instructor_id, $date, $start_time, $end_time) {
  $stmt = $conn->prepare(
    "SELECT 1 FROM InstructorAvailability
     WHERE instructor_id=?
       AND available_date=?
       AND NOT (end_time <= ? OR start_time >= ?)
     LIMIT 1"
  );
  $stmt->bind_param('isss', $instructor_id, $date, $start_time, $end_time);
  $stmt->execute();
  return (bool)$stmt->get_result()->fetch_row();
}

// Count how many instructors have overlapping availability on a date/time window
function concurrent_availability_count($conn, $date, $start_time, $end_time) {
  $stmt = $conn->prepare(
    "SELECT COUNT(DISTINCT instructor_id) AS c
     FROM InstructorAvailability
     WHERE available_date = ?
       AND is_active=1
       AND NOT (end_time <= ? OR start_time >= ?)"
  );
  $stmt->bind_param('sss', $date, $start_time, $end_time);
  $stmt->execute();
  $row = $stmt->get_result()->fetch_assoc();
  return $row ? (int)$row['c'] : 0;
}

// Any DriveSchedule collision in [start,end) for the instructor
function drive_collision_exists($conn, $instructor_id, $date, $start_time, $end_time) {
  $stmt = $conn->prepare(
    "SELECT 1 FROM DriveSchedule
     WHERE instructor_id=?
       AND drive_date=?
       AND scheduled_time >= ?
       AND scheduled_time <  ?
     LIMIT 1"
  );
  $stmt->bind_param('isss', $instructor_id, $date, $start_time, $end_time);
  $stmt->execute();
  return (bool)$stmt->get_result()->fetch_row();
}

// Any booked DriveSlot collision for the instructor
function slot_collision_exists($conn, $instructor_id, $window_start_dt, $window_end_dt) {
  $from = fmt($window_start_dt);
  $to   = fmt((clone $window_end_dt)->modify('-1 second')); // end exclusive
  $stmt = $conn->prepare(
    "SELECT 1 FROM DriveSlots
     WHERE instructor_id=?
       AND is_booked=1
       AND slot_datetime >= ?
       AND slot_datetime <= ?
     LIMIT 1"
  );
  $stmt->bind_param('iss', $instructor_id, $from, $to);
  $stmt->execute();
  return (bool)$stmt->get_result()->fetch_row();
}

// Vehicles down at a specific moment
function vehicles_down_count_at($conn, DateTime $dt) {
  $ts = $dt->format('Y-m-d H:i:s');
  $sql = "
    SELECT COUNT(DISTINCT vehicle_id) AS c
    FROM fleet_outages
    WHERE start_datetime <= ? AND end_datetime > ?";
  $stmt = $conn->prepare($sql);
  $stmt->bind_param('ss', $ts, $ts);
  $stmt->execute();
  $row = $stmt->get_result()->fetch_assoc();
  return $row ? (int)$row['c'] : 0;
}

// Active cars in fleet (status='active')
function active_cars_count($conn) {
  $rs = $conn->query("SELECT COUNT(*) AS c FROM fleet WHERE status='active'");
  $row = $rs ? $rs->fetch_assoc() : null;
  return $row ? (int)$row['c'] : 0;
}

// Capacity override helpers you already have (if you added them earlier)
// get_capacity_override_for_dt($conn, $dt)
// get_setting($conn, 'active_fleet_capacity')  // optional global cap

// Final capacity for an exact datetime (breakdowns INCLUDED)
function effective_capacity_for_dt($conn, DateTime $dt) {
  // 1) Per-slot override (time-window or all-day)
  if (function_exists('get_capacity_override_for_dt')) {
    $ov = get_capacity_override_for_dt($conn, $dt);
    if ($ov !== null && $ov >= 0) {
      // If you want overrides to win even if cars are down, return $ov here.
      // Better: clamp by physical availability:
      $active = active_cars_count($conn);
      $down   = vehicles_down_count_at($conn, $dt);
      return max(0, min($ov, $active - $down));
    }
  }

  // 2) Optional global cap
  $global = null;
  if (function_exists('get_setting')) {
    $sv = get_setting($conn, 'active_fleet_capacity');
    if ($sv !== null && ctype_digit((string)$sv)) $global = (int)$sv;
  }

  // 3) Physical availability minus outages
  $active = active_cars_count($conn);
  $down   = vehicles_down_count_at($conn, $dt);
  $physical = max(0, $active - $down);

  // If a global cap exists, respect the tighter of the two
  return ($global && $global > 0) ? min($physical, $global) : $physical;
}

function safe($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

// ---------- state / sticky vars ----------
$message = '';
$errors  = [];
$results = [];

$DEBUG = false;  // set to true to print counts under the tables

function dbg($label, $val, $DEBUG){
  if ($DEBUG) {
    echo '<div class="mono" style="color:#666;font-size:.9rem;margin:.25rem 0">'
       . safe($label) . ': ' . safe(is_scalar($val)? $val : json_encode($val)) . '</div>';
  }
}

// Sticky vars for GET/POST
$sel_create_instructor = isset($_POST['instructor_id']) ? (int)$_POST['instructor_id'] : 0;
$sel_view_instructor   = isset($_GET['view_instructor_id']) ? (int)$_GET['view_instructor_id'] : 0;
$sel_view_week         = $_GET['view_week'] ?? '';
if ($sel_view_week === '') {
  $monday = new DateTime('monday this week', $tz);
  $sel_view_week = $monday->format('Y-m-d');
}

    // Flash messages from redirects
    if (isset($_GET['created']) || isset($_GET['skipped'])) {
      $c = (int)($_GET['created'] ?? 0);
      $s = (int)($_GET['skipped'] ?? 0);
      $message = "Created $c availability block(s). Skipped $s due to conflicts.";
    }
    
    if (isset($_GET['fleet_saved'])) {
      $message = "Saved weekly fleet capacity (Mon–Sat).";
    }
    
    if (isset($_GET['msg'])) {
      $message = $_GET['msg'];
    }


if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'set_week_capacity') {
  $week_monday = $_POST['week_monday'] ?? '';
  $cap = max(0, (int)($_POST['week_capacity'] ?? 0)); // 0 = no cars (hard stop)
  if ($week_monday) {
    $m = new DateTime($week_monday, $tz);
    if ((int)$m->format('N') !== 1) { $m->modify('last monday'); }
    $end = (clone $m)->modify('+5 days'); // Mon–Sat

    for ($d = clone $m; $d <= $end; $d->modify('+1 day')) {
      upsert_capacity_override_all_day($conn, $d->format('Y-m-d'), $cap);
    }

    $qs = http_build_query([
      'view_instructor_id' => $sel_view_instructor ?: 0,
      'view_week'          => $m->format('Y-m-d'),
      'fleet_saved'        => 1,
    ]);
    header("Location: ".$_SERVER['PHP_SELF'].'?'.$qs);
    exit;
  }
}

// Update a single override row (all-day rows only)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'update_override') {
  $od  = $_POST['override_date'] ?? '';
  $cap = max(0, (int)($_POST['capacity'] ?? -1));

  if ($od && $cap >= 0) {
    $stmt = $conn->prepare("
      UPDATE fleet_capacity_overrides
      SET capacity = ?
      WHERE override_date = ? AND start_time IS NULL AND end_time IS NULL
      LIMIT 1
    ");
    $stmt->bind_param('is', $cap, $od);
    if ($stmt->execute()) {
      $message = "Updated override for $od.";
    } else {
      $errors[] = "Database error while updating override.";
    }
    $stmt->close();
    // After a successful update/delete, redirect:
    header('Location: ' . $_SERVER['PHP_SELF'] . '?msg=' . urlencode($message));
    exit;
  } else {
    $errors[] = "Invalid data for update.";
  }
}

// Delete a single override row (all-day rows only)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'delete_override') {
  $od = $_POST['override_date'] ?? '';
  if ($od) {
    $stmt = $conn->prepare("
      DELETE FROM fleet_capacity_overrides
      WHERE override_date = ? AND start_time IS NULL AND end_time IS NULL
      LIMIT 1
    ");
    $stmt->bind_param('s', $od);
    if ($stmt->execute()) {
      $message = "Removed override for $od.";
    } else {
      $errors[] = "Database error while deleting override.";
    }
    $stmt->close();
    // After a successful update/delete, redirect:
    header('Location: ' . $_SERVER['PHP_SELF'] . '?msg=' . urlencode($message));
    exit;
  } else {
    $errors[] = "Invalid date for delete.";
  }
}

// --- Capacity overrides (week/day or time-window) ---

// Does a column exist in the current DB?
function column_exists(mysqli $conn, string $table, string $col): bool {
  $sql = "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS
          WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?";
  $stmt = $conn->prepare($sql);
  $stmt->bind_param('ss', $table, $col);
  $stmt->execute();
  $stmt->bind_result($cnt);
  $stmt->fetch();
  $stmt->close();
  return ($cnt > 0);
}

// Create/replace an all-day override for a single date
function upsert_capacity_override_all_day($conn, string $date, int $capacity) {
  // Requires a UNIQUE key on (override_date, start_time, end_time) — see SQL below
  $stmt = $conn->prepare("
    INSERT INTO fleet_capacity_overrides (override_date, start_time, end_time, capacity)
    VALUES (?, NULL, NULL, ?)
    ON DUPLICATE KEY UPDATE capacity = VALUES(capacity)
  ");
  $stmt->bind_param('si', $date, $capacity);
  return $stmt->execute();
}

// Look up the most specific override for a given datetime
function get_capacity_override_for_dt($conn, DateTime $dt) {
  $date = $dt->format('Y-m-d');
  $time = $dt->format('H:i:s');
  $sql = "
    SELECT capacity,
           IFNULL(TIMESTAMPDIFF(MINUTE, start_time, end_time), 24*60) AS span
    FROM fleet_capacity_overrides
    WHERE override_date = ?
      AND (start_time IS NULL OR start_time <= ?)
      AND (end_time   IS NULL OR end_time   >  ?)
    ORDER BY (start_time IS NULL) ASC, span ASC
    LIMIT 1
  ";
  $stmt = $conn->prepare($sql);
  $stmt->bind_param('sss', $date, $time, $time);
  $stmt->execute();
  $row = $stmt->get_result()->fetch_assoc();
  return $row ? (int)$row['capacity'] : null;
}

define('MAX_STUDENTS_PER_SLOT', 3);

// ---------- POST: create availability (then redirect) ----------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'create') {
  $instructor_id = (int)($_POST['instructor_id'] ?? 0);
  $start_date    = $_POST['start_date'] ?? '';
  $end_date      = $_POST['end_date'] ?? '';
  $days          = $_POST['days'] ?? [];
  $daily_start   = $_POST['daily_start'] ?? '';
  $daily_end     = $_POST['daily_end'] ?? '';
  $slot_len      = max(15, (int)($_POST['slot_len'] ?? 60));
  $enforce_fleet = true; // always enforce

  if (!$instructor_id || !$start_date || !$end_date || !$daily_start || !$daily_end || empty($days)) {
    $errors[] = 'Missing required fields.';
  } else {
    $dayMap = ['MO'=>1,'TU'=>2,'WE'=>3,'TH'=>4,'FR'=>5,'SA'=>6,'SU'=>7];
    $wanted = [];
    foreach ((array)$days as $d) { if (isset($dayMap[$d])) { $wanted[] = $dayMap[$d]; } }

    $start = new DateTime($start_date, $tz);
    $end   = new DateTime($end_date,   $tz); $end->setTime(23,59,59);

    $ins = $conn->prepare(
      "INSERT INTO InstructorAvailability (instructor_id, available_date, start_time, end_time)
       VALUES (?,?,?,?)"
    );

    $created = 0;
    $skipped = 0;
    $active_fleet = $enforce_fleet ? get_active_fleet_count($conn) : null;

    for ($d = clone $start; $d <= $end; $d->modify('+1 day')) {
      if (!in_array((int)$d->format('N'), $wanted, true)) continue;

      $day_str = $d->format('Y-m-d');
      $dayStart = dt($day_str, $daily_start, $tz);
      $dayEnd   = dt($day_str, $daily_end,   $tz);
      if (!$dayStart || !$dayEnd || $dayEnd <= $dayStart) continue;

      for ($slotStart = clone $dayStart; $slotStart < $dayEnd; $slotStart->modify("+{$slot_len} minutes")) {
        $slotEnd = (clone $slotStart)->modify("+{$slot_len} minutes");
        if ($slotEnd > $dayEnd) break;

        $st_time = $slotStart->format('H:i:s');
        $et_time = $slotEnd->format('H:i:s');

        $conflicts = [
          'availability_overlap' => availability_overlap_exists($conn, $instructor_id, $day_str, $st_time, $et_time),
          'drive_collision'      => drive_collision_exists($conn, $instructor_id, $day_str, $st_time, $et_time),
          'slot_collision'       => slot_collision_exists($conn, $instructor_id, $slotStart, $slotEnd),
          'fleet_capacity'       => false,
        ];

        if ($enforce_fleet) {
  $availCount = concurrent_availability_count($conn, $day_str, $st_time, $et_time); // consider is_active=1 (see D)
  $driveCount = concurrent_drives_at($conn, $slotStart);
  $utilization = max($availCount, $driveCount); // booked drives take precedence

  $cap_for_slot = effective_capacity_for_dt($conn, $slotStart);
  if ($cap_for_slot > 0 && $utilization >= $cap_for_slot) {
    $conflicts['fleet_capacity'] = true; // skip creating this availability block
  }
}

        if (in_array(true, $conflicts, true)) {
          $skipped++;
          continue;
        }

        // Insert availability
        $ins->bind_param('isss', $instructor_id, $day_str, $st_time, $et_time);
        if ($ins->execute()) { $created++; } else { $skipped++; }
      }
    }

    // PRG redirect to keep selections and show message
    $monday = new DateTime($start_date, $tz);
    if ((int)$monday->format('N') !== 1) { $monday->modify('last monday'); }
    $qs = http_build_query([
      'view_instructor_id' => $instructor_id,
      'view_week'          => $monday->format('Y-m-d'),
      'created'            => $created,
      'skipped'            => $skipped,
    ]);
    header("Location: ".$_SERVER['PHP_SELF'].'?'.$qs);
    exit;
  }
}

// Range for ledger: this month → end of next month
$ledgerStart = (new DateTime('first day of this month', $tz))->format('Y-m-d');
$ledgerEnd   = (new DateTime('last day of next month',  $tz))->format('Y-m-d');

$overrides = [];
$stmt = $conn->prepare("
  SELECT override_date, capacity, start_time, end_time, created_at
  FROM fleet_capacity_overrides
  WHERE override_date BETWEEN ? AND ?
  ORDER BY override_date ASC, start_time ASC
");
$stmt->bind_param('ss', $ledgerStart, $ledgerEnd);
if ($stmt->execute()) {
  $res = $stmt->get_result();
  while ($r = $res->fetch_assoc()) $overrides[] = $r;
  $res->free();
}
$stmt->close();

// For context, also compute the default physical capacity per day (active fleet)
// You can get fancier later by subtracting outages per day.
$default_active_fleet = active_cars_count($conn);


// ---------- GET: data for selects ----------
$instructors = $conn->query("SELECT id, first_name, last_name FROM Instructors ORDER BY last_name, first_name");
?>
<!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <title>Admin Master Scheduler</title>
  <style>
  /* Metallic blue dark theme */
  :root{
    --bg1:#0a1930;      /* deep navy */
    --bg2:#0f2e5e;      /* metallic blue */
    --bg3:#1a437a;      /* steel blue */
    --text:#f5f9ff;     /* near-white */
    --muted:#a9c1e6;    /* muted text */
    --card-bg:rgba(255,255,255,.06);
    --card-border:rgba(255,255,255,.14);
    --accent:#58b1ff;   /* primary button + focus */
    --accent-2:#7cc6ff; /* hover */
    --danger:#ff6b7a;
    --table-stripe:rgba(255,255,255,.04);
    --table-hover:rgba(255,255,255,.08);
    --input-bg:#0f2142;
    --input-border:rgba(255,255,255,.18);
    --input-focus:#7cc6ff;
  }

  *{ box-sizing:border-box }
  html,body{ height:100% }
  body{
    font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,sans-serif;
    margin:24px;
    color:var(--text);
    background:
      radial-gradient(1200px 800px at 15% -10%, var(--bg3) 0%, transparent 60%),
      radial-gradient(1000px 700px at 110% 10%, var(--bg2) 0%, transparent 55%),
      linear-gradient(180deg, var(--bg1) 0%, #081223 100%);
    color-scheme: dark; /* better native form controls */
  }

  h1,h2,h3{ color:var(--text); margin:0 0 12px }
  h1{ letter-spacing:.2px }
  a{ color:var(--accent) }
  .mono{font-family:ui-monospace,SFMono-Regular,Menlo,Consolas,"Liberation Mono",monospace}

  /* Cards */
  .card{
    background:var(--card-bg);
    border:1px solid var(--card-border);
    border-radius:14px;
    box-shadow:0 10px 24px rgba(0,0,0,.25);
    padding:16px;
    margin:16px 0;
    backdrop-filter:saturate(120%) blur(6px);
  }

  /* Grid rows */
  .row{ display:grid; grid-template-columns:1fr 1fr; gap:16px }
  label{ display:block; margin:.5rem 0 .25rem; color:var(--muted); font-weight:600 }

  /* Inputs */
  input,select{
    width:100%;
    padding:.6rem .75rem;
    border-radius:10px;
    border:1px solid var(--input-border);
    background:var(--input-bg);
    color:var(--text);
    outline:none;
    transition:border-color .15s ease, box-shadow .15s ease, background .15s ease;
  }
  input::placeholder{ color:#9fb3d1 }
  input:focus,select:focus{
    border-color:var(--input-focus);
    box-shadow:0 0 0 3px color-mix(in oklab, var(--input-focus) 30%, transparent);
  }

  /* Pills (Mon–Sat) */
  .days{
    display:grid;
    grid-template-columns:repeat(6,minmax(0,1fr));
    gap:.5rem;
  }
  .day{ position:relative; cursor:pointer }
  .day input{ position:absolute; inset:0; opacity:0; pointer-events:none }
  .day span{
    display:block; text-align:center; padding:.5rem .75rem;
    border:1px solid var(--card-border);
    border-radius:9999px; font-weight:600; color:var(--text);
    background:linear-gradient(180deg, rgba(255,255,255,.05), rgba(255,255,255,.02));
    transition:all .15s ease;
    user-select:none;
  }
  .day:hover span{ border-color:var(--accent-2) }
  .day input:focus-visible + span{ outline:2px solid var(--accent); outline-offset:2px }
  .day input:checked + span{
    background:linear-gradient(180deg, var(--accent) 0%, #3da6ff 100%);
    color:#00132a; border-color:transparent;
  }

  /* Buttons */
  .btn{
    display:inline-block;
    background:linear-gradient(180deg, var(--accent) 0%, #3da6ff 100%);
    color:#00132a;
    border:0; border-radius:10px;
    padding:.65rem 1.05rem; cursor:pointer;
    font-weight:700; letter-spacing:.2px;
    box-shadow:0 6px 14px rgba(24,128,255,.35);
    transition:transform .06s ease, box-shadow .15s ease, filter .15s ease;
  }
  .btn:hover{ filter:brightness(1.05); box-shadow:0 8px 18px rgba(24,128,255,.45) }
  .btn:active{ transform:translateY(1px) }
  .btn[disabled]{ opacity:.6; cursor:not-allowed }

  /* Danger variant (inline Delete) */
  .btn[style*="background:#b00020"]{
    background:linear-gradient(180deg, #ff6b7a 0%, #ff4960 100%) !important;
    color:#2a0006 !important;
    box-shadow:0 6px 14px rgba(255,90,110,.35);
  }

  /* Tables */
  table{ width:100%; border-collapse:separate; border-spacing:0; margin-top:12px; overflow:hidden; border-radius:12px; }
  th,td{ padding:10px 12px; text-align:left; }
  thead th{
    background:linear-gradient(180deg, rgba(255,255,255,.10), rgba(255,255,255,.06));
    color:var(--text);
    border-bottom:1px solid var(--card-border);
    font-weight:700;
  }
  tbody td{ border-top:1px solid var(--card-border) }
  tbody tr:nth-child(even){ background:var(--table-stripe) }
  tbody tr:hover{ background:var(--table-hover) }

  /* Alerts */
  .ok{ color:#86ffd0 }
  .warn{ color:#ffd89b }
  .err{ color:#ff9aa8 }

  /* Back to dashboard link as button-looking */
  a[href*="admin-dashboard.php"]{
    background:linear-gradient(180deg, var(--accent) 0%, #3da6ff 100%);
    color:#00132a !important; text-decoration:none;
    border-radius:10px; padding:.6rem 1rem; font-weight:700;
    display:inline-block;
    box-shadow:0 6px 14px rgba(24,128,255,.35);
  }

  /* Small screens */
  @media (max-width:820px){
    .row{ grid-template-columns:1fr }
    body{ margin:16px }
  }
</style>

</head>
<body>
  <h1>Admin Master Scheduler</h1>

  <?php if (!empty($errors)): ?>
    <div class="card err">
      <strong>Errors:</strong>
      <ul><?php foreach ($errors as $e) echo '<li>'.safe($e).'</li>'; ?></ul>
    </div>
  <?php endif; ?>

  <?php if ($message): ?>
    <div class="card ok"><strong><?php echo safe($message); ?></strong></div>
  <?php endif; ?>
  
  <div class="card">
  <h2>Fleet Capacity for Loaded Week</h2>
  <form method="post" action="<?php echo safe($_SERVER['PHP_SELF']); ?>">
    <input type="hidden" name="action" value="set_week_capacity">
    <div class="row">
      <div>
        <label>Week (Monday)</label>
        <input type="date" name="week_monday" value="<?php echo safe($sel_view_week); ?>" required>
      </div>
      <div>
        <label>Capacity (cars available all day)</label>
        <input type="number" name="week_capacity" min="0" step="1" required>
      </div>
    </div>
    <button class="btn" style="margin-top:.5rem">Save Week Capacity</button>
  </form>
</div>

<div class="card">
  <h2>Fleet Capacity Overrides (This Month + Next)</h2>
  <div style="font-size:.95rem; color:#555; margin-bottom:.5rem;">
    Default active fleet (today): <strong><?php echo (int)$default_active_fleet; ?></strong>.
    Overrides below replace the default on their dates. All rows shown are all-day unless times appear.
  </div>

  <div style="overflow-x:auto;">
    <table>
      <thead>
        <tr>
          <th>Date</th>
          <th>Start</th>
          <th>End</th>
          <th>Capacity</th>
          <th>Quick Edit</th>
          <th>Remove</th>
        </tr>
      </thead>
      <tbody>
        <?php
        // Show “no rows” message if empty
        if (empty($overrides)) {
          echo '<tr><td colspan="6" style="color:#666;text-align:center;">No overrides saved for this window.</td></tr>';
        } else {
          foreach ($overrides as $r) {
            $od   = safe($r['override_date']);
            $st   = $r['start_time'] ? safe(substr($r['start_time'],0,5)) : '—';
            $et   = $r['end_time']   ? safe(substr($r['end_time'],0,5))   : '—';
            $cap  = (int)$r['capacity'];

            echo '<tr>';
            echo '<td>'.$od.'</td>';
            echo '<td>'.$st.'</td>';
            echo '<td>'.$et.'</td>';

            // Capacity display
            echo '<td>'.$cap.'</td>';

            // Quick edit only for all-day rows (the ones your week form creates)
            if ($r['start_time'] === null && $r['end_time'] === null) {
              echo '<td>
                      <form method="post" style="display:flex;gap:6px;align-items:center;">
                        <input type="hidden" name="action" value="update_override">
                        <input type="hidden" name="override_date" value="'.$od.'">
                        <input type="number" name="capacity" value="'.$cap.'" min="0" step="1" style="width:90px">
                        <button class="btn" style="padding:.35rem .65rem;">Update</button>
                      </form>
                    </td>';
              echo '<td>
                      <form method="post" onsubmit="return confirm(\'Remove override for '.$od.'?\')">
                        <input type="hidden" name="action" value="delete_override">
                        <input type="hidden" name="override_date" value="'.$od.'">
                        <button class="btn" style="background:#b00020;padding:.35rem .65rem;">Delete</button>
                      </form>
                    </td>';
            } else {
              // For time-window rows, show read-only for now
              echo '<td style="color:#666;">Time-window (read-only)</td>';
              echo '<td>—</td>';
            }

            echo '</tr>';
          }
        }
        ?>
      </tbody>
    </table>
  </div>
</div>


  <div class="card">
    <h2>Create Instructor Availability (Blocks)</h2>
    <form method="post" action="<?php echo safe($_SERVER['PHP_SELF']); ?>">
      <input type="hidden" name="action" value="create">

      <label>Instructor</label>
      <select name="instructor_id" required>
        <?php while ($i = $instructors->fetch_assoc()):
          $name = trim(($i['last_name'] ?? '').', '.($i['first_name'] ?? ''));
          $sel  = ($sel_create_instructor === (int)$i['id']) ? ' selected' : '';
        ?>
          <option value="<?php echo (int)$i['id']; ?>"<?php echo $sel; ?>>
            <?php echo safe($name ?: ('#'.$i['id'])); ?>
          </option>
        <?php endwhile; ?>
      </select>

      <div class="row">
        <div>
          <label>Start date</label>
          <input type="date" name="start_date" required>
        </div>
        <div>
          <label>End date</label>
          <input type="date" name="end_date" required>
        </div>
      </div>

      <label>Days of week</label>
      <div class="days">
        <label class="day"><input type="checkbox" name="days[]" value="MO"><span>Mon</span></label>
        <label class="day"><input type="checkbox" name="days[]" value="TU"><span>Tue</span></label>
        <label class="day"><input type="checkbox" name="days[]" value="WE"><span>Wed</span></label>
        <label class="day"><input type="checkbox" name="days[]" value="TH"><span>Thu</span></label>
        <label class="day"><input type="checkbox" name="days[]" value="FR"><span>Fri</span></label>
        <label class="day"><input type="checkbox" name="days[]" value="SA"><span>Sat</span></label>
      </div>

      <div class="row">
        <div>
          <label>Daily start time</label>
          <input type="time" name="daily_start" required value="08:00">
        </div>
        <div>
          <label>Daily end time</label>
          <input type="time" name="daily_end" required value="17:00">
        </div>
      </div>

      <div class="row">
        <div>
          <label>Block length (minutes)</label>
          <input type="number" name="slot_len" required min="30" step="15" value="60">
        </div>
      </div>

      <button class="btn">Create Availability</button>
    </form>
  </div>

  <div class="card">
    <h2>Weekly Snapshot (Conflicts at a glance)</h2>
    <?php $inst_rs = $conn->query("SELECT id, first_name, last_name FROM Instructors ORDER BY last_name, first_name"); ?>
    <form method="get" action="<?php echo safe($_SERVER['PHP_SELF']); ?>">
      <div class="row">
        <div>
          <label>Instructor</label>
          <select name="view_instructor_id" required>
            <?php while ($r = $inst_rs->fetch_assoc()):
              $id  = (int)$r['id'];
              $nm  = trim(($r['last_name'] ?? '').', '.($r['first_name'] ?? ''));
              $sel = ($id === $sel_view_instructor) ? ' selected' : '';
            ?>
              <option value="<?php echo $id; ?>"<?php echo $sel; ?>>
                <?php echo safe($nm ?: ('#'.$id)); ?>
              </option>
            <?php endwhile; ?>
          </select>
        </div>
        <div>
          <label>Week of (Monday)</label>
          <input type="date" name="view_week" value="<?php echo safe($sel_view_week); ?>" required>
        </div>
      </div>
      <button class="btn" style="margin-top:12px">Load</button>
    </form>
<!-- Back to Dashboard Button -->
<div style="margin:20px 0;">
    <a href="admin-dashboard.php" 
       style="display:inline-block; padding:10px 20px; background:#007bff; color:#fff; 
              text-decoration:none; border-radius:5px; font-weight:bold;">
        ← Back to Dashboard
    </a>
</div>

<?php
if (!empty($_GET['view_instructor_id']) && !empty($_GET['view_week'])) {
  $iid  = (int)$_GET['view_instructor_id'];
  $week = new DateTime($_GET['view_week'], $tz);
  if ((int)$week->format('N') !== 1) { $week->modify('last monday'); }
  $end = (clone $week)->modify('+6 days');

  $week_start = $week->format('Y-m-d');
  $week_end   = $end->format('Y-m-d');

  // Optional: how many students you want to allow per lesson visually
  $CAPACITY_PER_SLOT = 3;

  // Get instructor name for the header
  $nmStmt = $conn->prepare("SELECT first_name, last_name FROM Instructors WHERE id=?");
  $nmStmt->bind_param('i', $iid);
  $nmStmt->execute();
  $nmRes = $nmStmt->get_result();
  $nmRow = $nmRes ? $nmRes->fetch_assoc() : null;
  if ($nmRes) { $nmRes->free(); }
  $nmStmt->close();
  $instructor_name = $nmRow ? trim(($nmRow['last_name'] ?? '').', '.($nmRow['first_name'] ?? '')) : ('#'.$iid);

  // 1) preload minutes BEFORE the week (across ALL instructors)
  $prior = $conn->prepare("
    SELECT student_id, COALESCE(SUM(minutes),0) AS m
    FROM DriveSchedule
    WHERE drive_date < ?
    GROUP BY student_id
  ");
  $prior->bind_param('s', $week_start);
  $prior->execute();
  $prRes = $prior->get_result();
  $mins_to_date = [];
  while ($r = $prRes->fetch_assoc()) {
    $mins_to_date[(int)$r['student_id']] = (int)$r['m'];
  }
  if ($prRes) { $prRes->free(); }
  $prior->close();

  // 2) fetch all lessons for this instructor in the week
  $q = $conn->prepare("
    SELECT ds.id, ds.drive_date, ds.scheduled_time, ds.minutes,
           ds.student_id, s.first_name, s.last_name
    FROM DriveSchedule ds
    LEFT JOIN Students s ON s.id = ds.student_id
    WHERE ds.instructor_id = ?
      AND ds.drive_date BETWEEN ? AND ?
    ORDER BY ds.drive_date, ds.scheduled_time, ds.id
  ");
  $q->bind_param('iss', $iid, $week_start, $week_end);
  $q->execute();
  $res = $q->get_result();

  // 3) bucket rows by (date + time) and compute minutes-to-date on the fly
  $by_slot = [];           // key = "YYYY-MM-DD HH:MM:SS" (date + time)
  $order_keys = [];        // keeps chronological order of slots
  while ($row = $res->fetch_assoc()) {
    $slotKey = $row['drive_date'].' '.$row['scheduled_time'];
    $sid     = (int)$row['student_id'];
    $base    = $mins_to_date[$sid] ?? 0;   // minutes before THIS lesson
    $row['btw_to_date'] = $base;
    $mins_to_date[$sid] = $base + (int)$row['minutes'];

    if (!isset($by_slot[$slotKey])) { $by_slot[$slotKey] = []; $order_keys[] = $slotKey; }
    $by_slot[$slotKey][] = $row;
  }
  if ($res) { $res->free(); }
  $q->close();

  // 4) render
  echo '<h3 style="margin-top:18px">Schedule — '.safe($instructor_name).' (Week of '.$week->format('Y-m-d').')</h3>';
  echo '<table><thead>
          <tr><th>Date</th><th>Time</th><th>Student</th><th>BTW minutes to-date</th></tr>
        </thead><tbody>';

  if (empty($order_keys)) {
    echo '<tr><td colspan="4">— No scheduled drives this week —</td></tr>';
  } else {
    foreach ($order_keys as $key) {
      $rows = $by_slot[$key];
      // extract date/time for display
      [$d, $t] = explode(' ', $key, 2);
      $prettyDate = (new DateTime($d, $tz))->format('D M j, Y');
      $prettyTime = substr($t, 0, 5); // HH:MM
      $count      = count($rows);

      foreach ($rows as $idx => $r) {
        $nm = trim(($r['last_name'] ?? '').', '.($r['first_name'] ?? ''));
        if ($nm === ',') { $nm = 'Student #'.(int)$r['student_id']; }
        echo '<tr>';
        echo '<td>'.safe($prettyDate).'</td>';
        echo '<td>'.safe($prettyTime).(($idx===0) ? ' <span class="mono" style="color:#666">('.$count.'/'.$CAPACITY_PER_SLOT.')</span>' : '').'</td>';
        echo '<td>'.safe($nm).'</td>';
        echo '<td>'.(int)$r['btw_to_date'].'</td>';
        echo '</tr>';
      }
    }
  }

  echo '</tbody></table>';
}
?>
  </div>
</body>
</html>

