<?php
// admin-transcript.php
session_start();
require 'db.php';

function safe($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

$student = null;
$drives  = [];
$tot     = ['btw'=>0, 'ob'=>0];
$nos     = [];
$feeTotal = 0.00;

if (isset($_GET['student_id'])) {
  $sid = (int)$_GET['student_id'];

  // Student
  $s = $conn->prepare("
    SELECT id, first_name, last_name, email, permit_number, section_code, permit_received_at
    FROM Students WHERE id = ?
  ");
  $s->bind_param('i', $sid);
  $s->execute();
  $student = $s->get_result()->fetch_assoc();
  $s->close();

  if ($student) {
    // Drives (join lesson label)
    $q = $conn->prepare("
      SELECT ds.drive_date, ds.scheduled_time, ds.btw_minutes, ds.ob_minutes,
             ds.lesson_status, lg.label AS lesson_label
      FROM DriveSchedule ds
      LEFT JOIN lesson_groups lg ON lg.id = ds.lesson_group_id
      WHERE ds.student_id = ?
      ORDER BY ds.drive_date ASC, ds.scheduled_time ASC
    ");
    $q->bind_param('i', $sid);
    $q->execute();
    $dr = $q->get_result();
    while ($row = $dr->fetch_assoc()) {
      $row['btw_minutes'] = (int)$row['btw_minutes'];
      $row['ob_minutes']  = (int)$row['ob_minutes'];
      $tot['btw'] += $row['btw_minutes'];
      $tot['ob']  += $row['ob_minutes'];
      $drives[] = $row;
    }
    $q->close();

    // No-shows + fee total
    $ns = $conn->prepare("
      SELECT date, reason, fee
      FROM NoShows
      WHERE student_id = ?
      ORDER BY date ASC
    ");
    $ns->bind_param('i', $sid);
    $ns->execute();
    $r = $ns->get_result();
    while ($row = $r->fetch_assoc()) {
      $row['fee'] = is_null($row['fee']) ? null : (float)$row['fee'];
      if (!is_null($row['fee'])) $feeTotal += $row['fee'];
      $nos[] = $row;
    }
    $ns->close();
  }
}

function status_label($s) {
  $map = ['P'=>'Pass','F'=>'Fail','I'=>'Incomplete','N'=>'No-Show'];
  return $map[$s] ?? $s;
}

// Permit countdown (6-month window) if available
$days_remaining = null;
if ($student && !empty($student['permit_received_at'])) {
  $permit = new DateTime($student['permit_received_at']);
  $expiry = (clone $permit)->modify('+6 months');
  $today  = new DateTime();
  $diff   = $today->diff($expiry);
  $days_remaining = $diff->invert ? 0 : $diff->days;
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Printable Transcript</title>
  <style>
    :root{
      --bg1:#0a1930; --bg2:#0f2e5e; --fg:#f5f9ff; --card:#fff; --ink:#0a1930; --muted:#6b7c97; --accent:#58b1ff;
      --rule:#e6ecf5;
    }
    *{box-sizing:border-box}
    body{
      margin:24px; font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;
      color:var(--fg);
      background:linear-gradient(180deg,var(--bg1),var(--bg2));
    }
    .wrap{max-width:1024px;margin:0 auto}
    .nav{display:flex;justify-content:space-between;align-items:center;margin-bottom:14px}
    .btn{background:linear-gradient(180deg,var(--accent),#3da6ff); color:#00132a; padding:.55rem .9rem; border:none; border-radius:10px; font-weight:700; cursor:pointer; text-decoration:none; display:inline-block}
    .card{background:var(--card); color:var(--ink); border-radius:14px; padding:20px; box-shadow:0 8px 30px rgba(0,0,0,.25)}
    h1,h2{margin:.25rem 0}
    table{width:100%; border-collapse:collapse; margin-top:10px}
    th,td{padding:10px; border-bottom:1px solid var(--rule); vertical-align:top; font-size:14px}
    th{text-align:left; color:#274264; font-size:13px}
    .right{text-align:right}
    .muted{color:#6b7c97}
    .kpi{display:grid; grid-template-columns:repeat(4,1fr); gap:12px; margin-top:10px}
    .pill{background:#f6f9ff; border:1px solid #dce6f5; border-radius:12px; padding:10px}
    @media print{
      body{background:#fff; color:#000}
      .nav{display:none}
      .card{box-shadow:none;border:1px solid #ddd}
      a[href]:after{content:""}
    }
  </style>
</head>
<body>
<div class="wrap">
  <div class="nav">
    <a class="btn" href="admin-dashboard.php">← Admin Dashboard</a>
    <?php if ($student): ?>
      <button class="btn" onclick="window.print()">Print / Save PDF</button>
    <?php endif; ?>
  </div>

  <?php if(!$student): ?>
    <div class="card">
      <h1>Printable Transcript</h1>
      <p class="muted">Enter a Student ID to view and print the full transcript (drives, totals, no-shows).</p>
      <form method="get" style="display:flex;gap:8px;align-items:center;margin-top:10px">
        <input type="number" name="student_id" placeholder="Student ID" required
               style="padding:.6rem .8rem; border:1px solid #cfd8e3; border-radius:10px; flex:0 0 200px">
        <button class="btn">Open</button>
      </form>
    </div>
  <?php else: ?>
    <div class="card">
      <div style="display:flex; justify-content:space-between; align-items:flex-start; gap:16px; flex-wrap:wrap">
        <div>
          <h1><?= safe($student['first_name'].' '.$student['last_name']) ?></h1>
          <div class="muted"><?= safe($student['email'] ?: '') ?></div>
          <?php if(!empty($student['permit_number'])): ?>
            <div class="muted">Permit #: <?= safe($student['permit_number']) ?></div>
          <?php endif; ?>
          <?php if(!empty($student['section_code'])): ?>
            <div class="muted">Section: <?= safe($student['section_code']) ?></div>
          <?php endif; ?>
        </div>
        <div class="right">
          <?php if(!is_null($days_remaining)): ?>
            <div class="muted">Permit window remaining</div>
            <div style="font-weight:800; font-size:1.15rem"><?= (int)$days_remaining ?> days</div>
          <?php endif; ?>
        </div>
      </div>

      <div class="kpi">
        <div class="pill"><div class="muted">Total BTW Minutes</div><div style="font-weight:800"><?= (int)$tot['btw'] ?> min</div></div>
        <div class="pill"><div class="muted">Total OB Minutes</div><div style="font-weight:800"><?= (int)$tot['ob'] ?> min</div></div>
        <div class="pill"><div class="muted">BTW Clock Hours (55m)</div><div style="font-weight:800"><?= number_format($tot['btw']/55, 2) ?></div></div>
        <div class="pill"><div class="muted">OB Clock Hours (55m)</div><div style="font-weight:800"><?= number_format($tot['ob']/55, 2) ?></div></div>
      </div>

      <h2 style="margin-top:16px">Drive History</h2>
      <table>
        <thead>
          <tr>
            <th style="width:16%">Date</th>
            <th style="width:14%">Time</th>
            <th style="width:32%">Lesson</th>
            <th style="width:14%" class="right">BTW (min)</th>
            <th style="width:14%" class="right">OB (min)</th>
            <th style="width:10%">Status</th>
          </tr>
        </thead>
        <tbody>
        <?php if(empty($drives)): ?>
          <tr><td colspan="6" class="muted">No drives recorded.</td></tr>
        <?php else: foreach($drives as $d): ?>
          <tr>
            <td><?= safe(date('M j, Y', strtotime($d['drive_date']))) ?></td>
            <td><?= safe(date('g:i A', strtotime($d['scheduled_time']))) ?></td>
            <td><?= safe($d['lesson_label'] ?: '—') ?></td>
            <td class="right"><?= (int)$d['btw_minutes'] ?></td>
            <td class="right"><?= (int)$d['ob_minutes'] ?></td>
            <td><?= safe(status_label($d['lesson_status'])) ?></td>
          </tr>
        <?php endforeach; endif; ?>
        </tbody>
        <tfoot>
          <tr>
            <td></td><td></td>
            <td class="right" style="font-weight:800">Totals</td>
            <td class="right" style="font-weight:800"><?= (int)$tot['btw'] ?></td>
            <td class="right" style="font-weight:800"><?= (int)$tot['ob'] ?></td>
            <td></td>
          </tr>
        </tfoot>
      </table>

      <h2 style="margin-top:16px">No-Shows</h2>
      <table>
        <thead>
          <tr>
            <th style="width:20%">Date</th>
            <th style="width:60%">Reason</th>
            <th class="right" style="width:20%">Fee</th>
          </tr>
        </thead>
        <tbody>
          <?php if(empty($nos)): ?>
            <tr><td colspan="3" class="muted">No no-shows recorded.</td></tr>
          <?php else: foreach($nos as $n): ?>
            <tr>
              <td><?= safe(date('M j, Y', strtotime($n['date']))) ?></td>
              <td><?= $n['reason'] ? safe($n['reason']) : '—' ?></td>
              <td class="right"><?= is_null($n['fee']) ? '—' : '$'.number_format($n['fee'],2) ?></td>
            </tr>
          <?php endforeach; endif; ?>
        </tbody>
        <tfoot>
          <tr>
            <td></td>
            <td class="right" style="font-weight:800">Total Fees</td>
            <td class="right" style="font-weight:800">$<?= number_format($feeTotal,2) ?></td>
          </tr>
        </tfoot>
      </table>

      <p class="muted" style="margin-top:12px">
        Note: One “clock hour” equals 55 minutes for state reporting.
      </p>
    </div>
  <?php endif; ?>
</div>
</body>
</html>
