<?php
session_start();
require 'db.php';

// Redirect if not logged in
if (!isset($_SESSION['student_id'])) {
    header("Location: login.php");
    exit();
}

$student_id = (int)$_SESSION['student_id'];
$tz = new DateTimeZone('America/Chicago');

$success = "";
$error   = "";

// -------------------------- helpers --------------------------
function safe($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

// settings helper (optional global default capacity)
function get_setting($conn, $key) {
    if (!($stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key=? LIMIT 1"))) return null;
    $stmt->bind_param('s', $key);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();
    return $row ? $row['setting_value'] : null;
}

// count active cars in fleet table (fallback)
function active_cars_count($conn) {
    $rs = $conn->query("SELECT COUNT(*) AS c FROM fleet WHERE status='active'");
    $row = $rs ? $rs->fetch_assoc() : null;
    return $row ? (int)$row['c'] : 0;
}

// per-slot capacity override lookup (week/day/time), if table exists
function get_capacity_override_for_dt($conn, DateTime $dt) {
    // If table doesn't exist, this will fail — guard by checking once
    static $hasTable = null;
    if ($hasTable === null) {
        $check = $conn->query("SHOW TABLES LIKE 'fleet_capacity_overrides'");
        $hasTable = $check && $check->num_rows > 0;
    }
    if (!$hasTable) return null;

    $date = $dt->format('Y-m-d');
    $time = $dt->format('H:i:s');
    $sql = "
      SELECT capacity,
             IFNULL(TIMESTAMPDIFF(MINUTE, start_time, end_time), 24*60) AS span
      FROM fleet_capacity_overrides
      WHERE override_date = ?
        AND (start_time IS NULL OR start_time <= ?)
        AND (end_time   IS NULL OR end_time   >  ?)
      ORDER BY (start_time IS NULL) ASC, span ASC
      LIMIT 1";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('sss', $date, $time, $time);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();
    return $row ? (int)$row['capacity'] : null;
}

// number of vehicles down at a specific moment (if outages table exists)
function vehicles_down_count_at($conn, DateTime $dt) {
    static $hasTable = null;
    if ($hasTable === null) {
        $check = $conn->query("SHOW TABLES LIKE 'fleet_outages'");
        $hasTable = $check && $check->num_rows > 0;
    }
    if (!$hasTable) return 0;

    $ts = $dt->format('Y-m-d H:i:s');
    $stmt = $conn->prepare("
      SELECT COUNT(DISTINCT vehicle_id) AS c
      FROM fleet_outages
      WHERE start_datetime <= ? AND end_datetime > ?
    ");
    $stmt->bind_param('ss', $ts, $ts);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();
    return $row ? (int)$row['c'] : 0;
}

// final capacity for an exact datetime (override > global > physical), clamped by physical (active-down)
function effective_capacity_for_dt($conn, DateTime $dt) {
    $active  = active_cars_count($conn);
    $down    = vehicles_down_count_at($conn, $dt);
    $physical = max(0, $active - $down);

    // 1) time-window or all-day override for this date/time
    $ov = get_capacity_override_for_dt($conn, $dt);
    if ($ov !== null && $ov >= 0) {
        return max(0, min($ov, $physical)); // clamp to physical
    }

    // 2) optional global setting
    $sv = get_setting($conn, 'active_fleet_capacity');
    if ($sv !== null && ctype_digit((string)$sv) && (int)$sv > 0) {
        return max(0, min((int)$sv, $physical)); // clamp to physical
    }

    // 3) otherwise physical capacity
    return $physical;
}

// Count actual student bookings at a specific minute using BookedDrives ↔ InstructorAvailability
function booked_drives_at($conn, DateTime $dt) {
    $date = $dt->format('Y-m-d');
    $time = $dt->format('H:i:s');
    $sql = "
      SELECT COUNT(*) AS c
      FROM BookedDrives bd
      JOIN InstructorAvailability ia ON ia.id = bd.availability_id
      WHERE ia.available_date = ?
        AND ia.start_time    = ?
        AND ia.is_active     = 1
    ";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('ss', $date, $time);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();
    return $row ? (int)$row['c'] : 0;
}

// -------------------------- POST: booking --------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $availability_id = (int)($_POST['availability_id'] ?? 0);

    // 1) Basic student gating
    $stmt = $conn->prepare("SELECT permit_number, classroom_completed FROM Students WHERE id=?");
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();

    if (empty($student['permit_number']) || !(int)$student['classroom_completed']) {
        $error = "You must have a permit and complete the classroom course before booking a drive.";
    }

    // 2) Load the selected slot details
    if (!$error) {
        if ($availability_id > 0) {
            $q = $conn->prepare("
              SELECT ia.available_date, ia.start_time, ia.end_time
              FROM InstructorAvailability ia
              WHERE ia.id=? AND ia.is_active=1
              LIMIT 1
            ");
            $q->bind_param("i", $availability_id);
            $q->execute();
            $slot = $q->get_result()->fetch_assoc();
            if (!$slot) {
                $error = "That slot is no longer available.";
            } else {
                $when = DateTime::createFromFormat('Y-m-d H:i:s', $slot['available_date'].' '.$slot['start_time'], $tz);
                if (!$when) {
                    $error = "Invalid slot time.";
                }
            }
        } else {
            $error = "No slot selected.";
        }
    }

    // 3) Capacity & atomic booking (advisory lock to prevent races)
    if (!$error) {
        // lock per minute (adjust granularity if your slots are longer/shorter)
        $lockName = 'cap:'.$when->format('Y-m-d H:i');
        $lockStmt = $conn->prepare("SELECT GET_LOCK(?, 5)");
        $lockStmt->bind_param('s', $lockName);
        $lockStmt->execute();
        list($gotLock) = $lockStmt->get_result()->fetch_row();
        if ((int)$gotLock !== 1) {
            $error = "System is busy. Please try again.";
        } else {
            $txStarted = false;
            try {
                // Slot already taken?
                $stmt = $conn->prepare("SELECT 1 FROM BookedDrives WHERE availability_id=? LIMIT 1");
                $stmt->bind_param("i", $availability_id);
                $stmt->execute();
                if ($stmt->get_result()->fetch_row()) {
                    throw new Exception("This slot has already been booked.");
                }

                // Capacity gate: use real booking

