<?php
session_start();
require 'db.php';

if (!isset($_SESSION['instructor_id'])) {
    header("Location: instructor-login.php");
    exit();
}

$instructor_id = $_SESSION['instructor_id'];
$instructor_name = $_SESSION['instructor_name'] ?? 'Instructor';

$today = date('Y-m-d');
$ten_days_later = date('Y-m-d', strtotime('+10 days'));

// Upcoming drives query

$stmt = $conn->prepare("
    SELECT 
        ia.available_date AS drive_date,
        ia.start_time AS scheduled_time,
        s.first_name AS student_first,
        s.last_name AS student_last,
        IFNULL(totals.total_btw, 0) AS total_btw,
        IFNULL(totals.total_ob, 0) AS total_ob
    FROM BookedDrives bd
    JOIN InstructorAvailability ia ON bd.slot_id = ia.id
    JOIN Students s ON bd.student_id = s.id
    LEFT JOIN (
        SELECT student_id,
               SUM(COALESCE(btw_minutes, 0)) AS total_btw,
               SUM(COALESCE(ob_minutes, 0)) AS total_ob
        FROM DriveSchedule
        WHERE drive_date < CURDATE()
        GROUP BY student_id
    ) AS totals ON totals.student_id = bd.student_id
    WHERE ia.instructor_id = ?
      AND DATE(ia.available_date) BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 10 DAY)
    ORDER BY ia.available_date, ia.start_time
");

if (!$stmt) {
    die("Prepare failed: " . $conn->error);
}

$stmt->bind_param("i", $instructor_id);
$stmt->execute();
$drives = $stmt->get_result();
$stmt->close();

    if (!$stmt) {
    die("Prepare failed: " . $conn->error);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Instructor Dashboard</title>
    <style>
    
    body {
        font-family: Arial, sans-serif;
        background: url('hotrod-1.jpg') no-repeat center fixed;
        background-size: cover;
        margin: 0;
        padding: 0;
        min-height: 100vh;
}


    .container {
        max-width: 900px;
        margin: auto;
        background: #fff;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 0 10px rgba(0,0,0,0.1);
        text-align: center;
        transform: translateY(300px);
    }

    h2 {
        margin-bottom: 25px;
    }

    .button-group {
        display: flex;
        flex-wrap: wrap;
        justify-content: center;
        gap: 10px;
        margin-bottom: 20px;
    }

    .button-group a {
        padding: 12px 18px;
        text-decoration: none;
        background-color: #007bff;
        color: white;
        border-radius: 5px;
        flex: 1 1 180px;
        max-width: 200px;
        text-align: center;
    }

    .button-group.logout a {
        background-color: #dc3545;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
        text-align: left;
    }

    th, td {
        border: 1px solid #ccc;
        padding: 10px;
    }

    th {
        background-color: #f0f0f0;
    }

    .no-drives {
        text-align: center;
        color: gray;
        margin-top: 20px;
    }

    @media (max-width: 600px) {
        .button-group {
            flex-direction: column;
        }

        .button-group a {
            width: 100%;
            max-width: none;
        }

        table, th, td {
            font-size: 14px;
        }
    }
</style>

<body>
<div class="container">
  <h2>Welcome, <?= htmlspecialchars($instructor_name); ?> – Upcoming Drives (Next 10 Days)</h2>

  <div class="button-wrapper">
    <div class="button-group">
        <a href="instructor-drive-booking.php">Book a Student Drive</a>
        <a href="log-drive.php">Log Student Drive</a>
        <a href="log-noshow.php">Log No-Show</a>
        <a href="instructor-availability.php">Submit Availability</a>
        <a href="lookup-student.php">Look Up Student</a>
    </div>
    <div class="button-group logout">
        <a href="instructor-login.php">Logout</a>
    </div>
  </div>

  <?php if ($drives->num_rows > 0): ?>
    <table>
      <thead>
        <tr>
          <th>Student</th>
          <th>Date | Time</th>
          <th>Minutes</th>
        </tr>
      </thead>
      <tbody>
        <?php while ($row = $drives->fetch_assoc()): ?>
        <tr>
          <td><?= htmlspecialchars($row['student_first'] . ' ' . $row['student_last']) ?></td>
          <td><?= htmlspecialchars($row['drive_date']) ?> |
              <?= isset($row['scheduled_time']) ? date('g:i A', strtotime($row['scheduled_time'])) : 'N/A' ?>
          </td>
          <td>BTW <?= (int)$row['total_btw'] ?> | OB <?= (int)$row['total_ob'] ?></td>
        </tr>
        <?php endwhile; ?>
      </tbody>
    </table>
  <?php else: ?>
    <p class="no-drives">No scheduled drives for the next 10 days.</p>
  <?php endif; ?>
</div>
</body>
</html>


