<?php
session_start();
require 'db.php';

$results = [];
$error = '';

if ($_SERVER["REQUEST_METHOD"] === "POST" && isset($_POST['search'])) {
    $query = trim($_POST['search']);
    $like = "%$query%";

    $stmt = $conn->prepare("
        SELECT * FROM Students 
        WHERE 
            first_name LIKE ? OR 
            last_name LIKE ? OR 
            CONCAT(first_name, ' ', last_name) LIKE ?
    ");
    $stmt->bind_param("sss", $like, $like, $like);
    $stmt->execute();
    $students = $stmt->get_result();

    while ($student = $students->fetch_assoc()) {
        $student_id = $student['id'];
        
        // No-show list (dates/reasons) + total fees
        $nsStmt = $conn->prepare("
            SELECT date, fee, reason
            FROM NoShows
            WHERE student_id = ?
            ORDER BY date ASC
        ");
        $nsStmt->bind_param("i", $student_id);
        $nsStmt->execute();
        $noShows = $nsStmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $nsStmt->close();
        
        $feeStmt = $conn->prepare("SELECT COALESCE(SUM(fee),0) AS total_fee FROM NoShows WHERE student_id = ?");
        $feeStmt->bind_param("i", $student_id);
        $feeStmt->execute();
        $feeTotal = $feeStmt->get_result()->fetch_assoc()['total_fee'];
        $feeStmt->close();
        
        $student['no_shows']   = $noShows;
        $student['fee_total']  = (float)$feeTotal;

        // Calculate minutes
        $minutesStmt = $conn->prepare("
            SELECT 
                COALESCE(SUM(btw_minutes), 0) AS total_btw,
                COALESCE(SUM(ob_minutes), 0) AS total_ob
            FROM DriveSchedule
            WHERE student_id = ?
        ");
        $minutesStmt->bind_param("i", $student_id);
        $minutesStmt->execute();
        $mins = $minutesStmt->get_result()->fetch_assoc();
        $minutesStmt->close();
        // No-shows: list + total fees
        $nsStmt = $conn->prepare("
            SELECT date, fee, reason
            FROM `NoShows`
            WHERE student_id = ?
            ORDER BY date DESC
        ");
        $nsStmt->bind_param("i", $student_id);
        $nsStmt->execute();
        $nsRes = $nsStmt->get_result();
        
        $no_shows = [];
        $total_fee = 0.0;
        
        while ($row = $nsRes->fetch_assoc()) {
            // accumulate total (ignore NULL fees)
            if ($row['fee'] !== null) {
                $total_fee += (float)$row['fee'];
            }
            $no_shows[] = [
                'date'   => $row['date'],
                'fee'    => $row['fee'],
                'reason' => $row['reason'] ?? ''
            ];
        }
        $nsStmt->close();
        
        $student['total_no_show_fees'] = $total_fee;
        $student['no_shows'] = $no_shows;
        $student['total_btw'] = $mins['total_btw'];
        $student['total_ob'] = $mins['total_ob'];

        // Countdown logic
        if (!empty($student['permit_received_at'])) {
            $permit_date = new DateTime($student['permit_received_at']);
            $expiry_date = (clone $permit_date)->modify('+6 months');
            $today = new DateTime();
            $interval = $today->diff($expiry_date);
            $student['days_remaining'] = $interval->invert ? 0 : $interval->days;
        } else {
            $student['days_remaining'] = 'N/A';
        }

        $results[] = $student;
    }

    $stmt->close();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Lookup Student</title>
    <link rel="stylesheet" href="styles.css">
    <style>
        .container {
            max-width: 600px;
            margin: 150px auto;
            padding: 25px;
            border-radius: 10px;
            background: rgba(255,255,255,0.90);
            box-shadow: 0 0 15px rgba(0,0,0,0.1);
        }
        input[type="text"], input[type="submit"] {
            width: 100%;
            padding: 12px;
            margin-bottom: 10px;
        }
        .student {
            background: #f9f9f9;
            padding: 15px;
            margin-top: 20px;
            border-radius: 8px;
        }
        h2 { text-align: center; }
    </style>
</head>
<body>
<div class="container">
    <h2>Lookup Student</h2>

    <form method="post">
        <input type="text" name="search" placeholder="Enter full or partial name..." required>
        <input type="submit" value="Search">
    </form>

    <?php if ($_SERVER["REQUEST_METHOD"] === "POST" && empty($results)): ?>
        <p style="color: red;">No matching students found.</p>
    <?php endif; ?>

    <?php foreach ($results as $student): ?>
        <div class="student">
            <p><strong>Name:</strong> <?= htmlspecialchars($student['first_name'] . ' ' . $student['last_name']) ?></p>
            <p><strong>Email:</strong> <?= htmlspecialchars($student['email']) ?></p>
            <p><strong>Section Code:</strong> <?= htmlspecialchars($student['section_code'] ?: 'N/A') ?></p>
            <p><strong>Classroom Completed:</strong> <?= $student['classroom_completed'] ? 'Yes' : 'No' ?></p>
            <p><strong>Permit Number:</strong> <?= htmlspecialchars($student['permit_number'] ?: 'Not provided') ?></p>
            <p><strong>Days Remaining:</strong> <?= is_numeric($student['days_remaining']) ? $student['days_remaining'] . ' days' : 'N/A' ?></p>
            <p><strong>DE-964:</strong> <?= $student['de964_number'] ?: 'N/A' ?></p>
            <p><strong>ADE-1317:</strong> <?= $student['ade1317_number'] ?: 'N/A' ?></p>
            <p><strong>BTW Minutes:</strong> <?= $student['total_btw'] ?> mins</p>
            <p><strong>Observation Minutes:</strong> <?= $student['total_ob'] ?> mins</p>
            <?php
              // tiny helper to keep consistent escaping
              function safe($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
            ?>
            <p><strong>No-Show Fees Total:</strong>
               $<?= number_format($student['fee_total'], 2) ?></p>
            
            <?php if (!empty($student['no_shows'])): ?>
              <div style="margin-top:8px">
                <strong>No-Show Log:</strong>
                <ul style="margin:.35rem 0 .1rem .95rem; padding:0">
                  <?php foreach ($student['no_shows'] as $ns): ?>
                    <li>
                      <?= safe(date('M j, Y', strtotime($ns['date']))) ?>
                      <?= $ns['reason'] ? ' — '.safe($ns['reason']) : '' ?>
                      <?= is_null($ns['fee']) ? '' : ' ($'.number_format((float)$ns['fee'],2).')' ?>
                    </li>
                  <?php endforeach; ?>
                </ul>
              </div>
            <?php else: ?>
              <p class="muted" style="color:#777">No no-shows recorded.</p>
            <?php endif; ?>
            
            <div style="display:flex; gap:8px; flex-wrap:wrap; margin-top:10px">
              <a href="admin-invoice.php?student_id=<?= (int)$student['id'] ?>"
                 style="text-decoration:none; padding:.55rem .9rem; border-radius:8px; background:#0f2e5e; color:#fff; font-weight:700">
                 View Invoice
              </a>
              <a href="admin-transcript.php?student_id=<?= (int)$student['id'] ?>"
                 style="text-decoration:none; padding:.55rem .9rem; border-radius:8px; background:#58b1ff; color:#00132a; font-weight:700">
                 Print Transcript
              </a>
            </div>


            <?php if (!empty($student['no_shows'])): ?>
            <div class="no-shows">
                <strong>No-Show Dates:</strong>
                <ul style="margin:8px 0 0 18px;">
                    <?php foreach ($student['no_shows'] as $ns): ?>
                        <li>
                            <?= htmlspecialchars(date('M j, Y', strtotime($ns['date']))) ?>
                            <?php if ($ns['fee'] !== null): ?>
                                — $<?= number_format((float)$ns['fee'], 2) ?>
                            <?php endif; ?>
                            <?php if (!empty($ns['reason'])): ?>
                                — <?= htmlspecialchars($ns['reason']) ?>
                            <?php endif; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php else: ?>
            <p><strong>No-Show Dates:</strong> None</p>
        <?php endif; ?>
        </div>
    <?php endforeach; ?>
    <form action="admin-dashboard.php" method="get" style="text-align: center; margin-top: 30px;">
    <input type="submit" value="Return to Dashboard" style="padding: 10px 20px; background-color: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer;">
</form>

</div>

</body>
</html>





