<?php
require 'db.php';

$availability = [];
$today = new DateTime();
$daysChecked = 0;

while (count($availability) < 14 && $daysChecked < 21) {
    $currentDay = clone $today;
    $currentDay->modify("+$daysChecked days");
    $weekday = $currentDay->format('N'); // 1 = Monday, 7 = Sunday

    if ($weekday < 6) { // Weekdays only
        $dateStr = $currentDay->format('Y-m-d');
        $timeSlots = ['08:30:00', '08:40:00', '08:50:00', '09:00:00', '09:10:00', '09:20:00', '09:30:00'];

        $placeholders = implode(',', array_fill(0, count($timeSlots), '?'));
        $types = str_repeat('s', count($timeSlots) + 1);
        $params = array_merge([$dateStr], $timeSlots);

        $stmt = $conn->prepare("SELECT test_time FROM TestAppointments WHERE test_date = ? AND test_time IN ($placeholders)");
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $result = $stmt->get_result();

        $booked = [];
        while ($row = $result->fetch_assoc()) {
            $booked[] = $row['test_time'];
        }

        $availableSlots = array_diff($timeSlots, $booked);

        if (!empty($availableSlots)) {
            $availability[] = [
                'date' => $currentDay->format('D, M j'),
                'slots' => $availableSlots
            ];
        }
    }

    $daysChecked++;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Schedule Driving Test</title>
    <link rel="stylesheet" href="../styles.css">
    <script src="https://www.google.com/recaptcha/api.js" async defer></script>
    <style>
        body {
            font-family: Arial, sans-serif;
            padding: 30px;
        }
        .container {
            max-width: 600px;
            margin: auto;
            background: rgba(255,255,255,0.85);
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
            transform: translateY(150px);
        }
        h2 {
            text-align: center;
        }
        h1 {text align: center;
        }
        
        label {
            margin-top: 10px;
            display: block;
            font-weight: bold;
        }
        input, select {
            width: 100%;
            padding: 10px;
            margin-top: 4px;
            margin-bottom: 15px;
            border-radius: 5px;
            border: 1px solid #ccc;
        }
        .submit-btn {
            background: #007bff;
            color: #fff;
            font-weight: bold;
            border: none;
            padding: 12px;
            border-radius: 5px;
            cursor: pointer;
            width: 100%;
        }
        .submit-btn:hover {
            background: #0056b3;
        }
    </style>
</head>
<body>
    <div class="container">
        <div style="display: flex; gap: 30px;">
    <div style="flex: 2;">
        <!-- Original form remains here -->
        <h1>Schedule Your Driving Test</h1>
        <h2>Tests can be scheduled 14 days out</h2>
        <form action="schedule-test.php" method="POST">
            <label for="first_name">First Name:</label>
            <input type="text" name="first_name" id="first_name" required>

            <label for="last_name">Last Name:</label>
            <input type="text" name="last_name" id="last_name" required>

            <label for="email">Email Address:</label>
            <input type="email" name="email" id="email" required>

            <label for="phone">Phone Number:</label>
            <input type="text" name="phone" id="phone" required>

            <label for="dob">Date of Birth:</label>
            <input type="date" name="dob" id="dob" required>

            <label for="test_date">Choose Test Date:</label>
            <input type="date" name="test_date" required
                min="<?= date('Y-m-d'); ?>"
                max="<?= date('Y-m-d', strtotime('+14 days')); ?>">

            <label for="test_time">Choose Arrival Time (Only 7 available before 9:30am):</label>
            <select name="test_time" id="test_time" required>
                <option value="">-- Select Time --</option>
                <option value="8:30">8:30 AM</option>
                <option value="8:40">8:40 AM</option>
                <option value="8:50">8:50 AM</option>
                <option value="9:00">9:00 AM</option>
                <option value="9:10">9:10 AM</option>
                <option value="9:20">9:20 AM</option>
                <option value="9:30">9:30 AM</option>
                <option value="10:00">10:00 AM ($125 late fee applies)</option>
                <option value="11:00">11:00 AM ($125 late fee applies)</option>
            </select>

            <!-- ✅ This is your working reCAPTCHA -->
            <div class="g-recaptcha" data-sitekey="6LenHlgrAAAAAGRgqcY3RQJX8XcZhhs95F55XFov"></div>

            <br>
            <button type="submit" class="submit-btn">Submit Appointment</button>
        </form>
            </div>
        <div style="flex: 1; background: rgba(255,255,255,0.85); padding: 15px; border-radius: 10px; box-shadow: 0 0 8px rgba(0,0,0,0.1); max-height: 500px; overflow-y: auto;">
            <h3 style="text-align: center;">Available Slots</h3>
            <?php if (empty($availability)): ?>
                <p>No weekday slots currently available.</p>
            <?php else: ?>
                <table style="width: 100%; border-collapse: collapse;">
                    <?php foreach ($availability as $day): ?>
                        <tr>
                            <td style="font-weight: bold; padding-top: 10px;"><?= $day['date']; ?></td>
                        </tr>
                        <tr>
                            <td style="padding-bottom: 10px;">
                                <?php foreach ($day['slots'] as $slot): ?>
                                    <div style="margin: 2px 0;"><?= date('g:i A', strtotime($slot)); ?></div>
                                <?php endforeach; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </table>
            <?php endif; ?>
        </div>
    </div>
    </div>
    <script src="js/tpst.js"></script>
</body>
</html>

